package org.lsst.ccs.subsystems.shutter;

//import org.lsst.ccs.subsystems.shutter.*;

import org.lsst.ccs.subsystems.shutter.status.HallTransitionStatus;
import org.lsst.ccs.subsystems.shutter.status.MovementStatus;
import org.lsst.ccs.subsystems.shutter.status.ShutterStatus;
import org.lsst.ccs.framework.Module;
//import org.lsst.ccs.state.State;
//import org.lsst.ccs.subsystems.shutter.interfaces.BladePosition;
import org.lsst.ccs.subsystems.shutter.interfaces.HallTransition;
import org.lsst.ccs.subsystems.shutter.interfaces.MovementHistory;
//import org.lsst.ccs.subsystems.shutter.interfaces.BladeSet;
import org.lsst.ccs.subsystems.shutter.status.MoveBladeSetStatus;
import org.lsst.ccs.subsystems.shutter.status.TakeImageStatus;
import org.lsst.ccs.subsystems.shutter.status.ReadyForActionStatus;
//import org.lsst.ccs.subsystems.shutter.status.ShutterStatus;
import org.lsst.ccs.subsystems.shutter.status.CloseShutterStatus;
//import org.lsst.ccs.subsystems.shutter.common.HallTransitionImpl;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.Timer;
//import javax.swing.event.ChangeEvent;
//import javax.swing.event.ChangeListener;
//import javax.swing.event.EventListenerList;

import java.util.List;
import org.lsst.ccs.subsystems.shutter.interfaces.BladeSet;
import org.lsst.ccs.subsystems.shutter.interfaces.MovementHistoryListener;

/**
 *
 * @author azemoon
 */
public class ShutterModule extends Module {

    static final double MAXCLOSETIME = 2;

    private float moveTimeSeconds = 2;
    private float exposureTimeSeconds = 10;
    private List<BladeSet> bladeSets = null;
    private boolean moveSingleBlade = false;

    public void setBladeSets(List<BladeSet> bladeSets) {
        this.bladeSets = bladeSets;
    }

    public BladeSet getBladeSet(int index) {
        return bladeSets.get(index);
    }

    private void logFatalMessageAndThrowException(String msg) {
        log.fatal(msg);
        throw new RuntimeException(msg);
    }

    @Override
    public void initModule() {
        log.info("[ShutterModule] Initializing the Shutter module ");

        // Check that we have two BladeSets
        if (bladeSets.size() != 2) {
            logFatalMessageAndThrowException("[ShutterModule] The Shutter Module requires two bladeSets when initialized");
        }

        // Check that the bladeSets have been registerd correctly in
        // the xml configuration file.
        if (getBladeSet(0).getIndex() != 0 || getBladeSet(1).getIndex() != 1) {
            logFatalMessageAndThrowException("[ShutterModule] At least one BladeSet has been registered with the wrong index");
        }
        
        // TODO do the hardware initialization somewhere

        // CHECK ON THE BLADES AND FIGURE OUT WHERE THEY ARE.
        // SET THE STATE OF THE SHUTTER
        double bladeSetPosition0 = getBladeSetPosition(0);
        double bladeSetPosition1 = getBladeSetPosition(1);

        // TODO do not test to 0. or 1.  but include mechanical tolerance.
        /**
        if ((bladeSetPosition0 == 0. && bladeSetPosition1 == 1.)
                || (bladeSetPosition0 == 1. && bladeSetPosition1 == 0.)) {
            mode = READY_FOR_IMAGES;
        } else {
            mode = NOT_READY_FOR_IMAGES;
        }

         */
    }

    /**
     * What has to be done for each tick of the timer.
     * We have to notify our observers we have changed
     * and give them the new values of the update data.
     */
    @Override
    public void tick() {
        try {
            notifyStatusOnBus();
        } catch (Exception e) {
            log.error(e);
        }
    }

    private void notifyStatusOnBus() {
        try {
            if (getContext() != null) {
                sendToStatus(new ShutterStatus(this));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * This time should on average be equal to the effective exposure time
     * Any asymmetry, offset, should be calibrated and handled by the implementation
     * @return the exposure time
     */
    
    public double getExposureTimeSeconds() {
        return exposureTimeSeconds;
    }

    public void setExposureTimeSeconds(double exposureTimeSeconds) {
        log.info("[ShutterModule] Setting default exposure time to  " + exposureTimeSeconds + " seconds.");
        this.exposureTimeSeconds = (float)exposureTimeSeconds;
        notifyStatusOnBus();
    }

    public double getMoveTimeSeconds() {
        return moveTimeSeconds;
    }

    public void setMoveTimeSeconds(float moveTimeSeconds) {
        log.info("[ShutterModule] Setting default open time to  " + moveTimeSeconds + " seconds.");
        this.moveTimeSeconds = moveTimeSeconds;
        notifyStatusOnBus();
    }

    public double getBladeSetPosition(int index) {
        return getBladeSet(index).getCurrentPosition();
    }

    public void moveToPosition(final int index, float targetPosition, final float moveTimeSeconds, final boolean initialize) {
        final BladeSet bladeSet = bladeSets.get(index);
        double currentPosition = getBladeSetPosition(index);
        //System.out.println("UI: currentPosition " + currentPosition + "targetPosition  " + targetPosition );
        boolean retractBladeSet = targetPosition < currentPosition;

        if (!retractBladeSet && initialize){
            double safeTargetPosition = 1 - getBladeSetPosition(1 - index);
            if ( targetPosition > safeTargetPosition ) {
                log.info("[ShutterModule] Unsafe operation, target position modified");
                System.out.println("********* Unsafe operation, target position modified ******* ");
                targetPosition = (float) safeTargetPosition;
                log.info("[ShutterModule] currentPosition " + currentPosition + "targetPosition  " + targetPosition );
                System.out.println("ShutterModule: currentPosition " + currentPosition + "targetPosition  " + targetPosition );
            }
        }

        bladeSet.addMovementHistoryListener(new MovementHistoryListener() {
            @Override
            public void movementHistoryFinalized(MovementHistory history) {
                System.out.println("ShutterModule: movementHistoryFinalized profile size " + history.getMovementProfile().size());
                System.out.println("ShutterModule: movementHistoryFinalized hall tr size " + history.getHallTransitions().size());
                System.out.println("ShutterModule: movementHistoryFinalized status " + history.getStatus() );
                sendToStatus(new MovementStatus(index, true, history));
                if (! initialize || moveSingleBlade) {
                    sendToStatus(new ReadyForActionStatus(true));
                    moveSingleBlade = false;
                }
                bladeSet.removeMovementHistoryListener(this);
            }

            @Override
            public void movementHistoryUpdated(MovementHistory history) {
                System.out.println("ShutterModule: movementHistoryUpdated");
                int lastIndex = history.getHallTransitions().size() - 1;
                HallTransition transition = history.getHallTransitions().get(lastIndex);
                sendToStatus(new HallTransitionStatus (index, transition));
            }
        });

        if (moveSingleBlade) sendToStatus(new MoveBladeSetStatus(bladeSet, (float) targetPosition, moveTimeSeconds));
        bladeSet.moveToPosition(targetPosition, moveTimeSeconds);
    }

    public void moveToPosition(final int index, float targetPosition, final float moveTimeSeconds) {
        moveSingleBlade = true;
        moveToPosition(index, targetPosition, moveTimeSeconds, true);
    }

    public void takeImage()
            throws InterruptedException, IllegalArgumentException {

        takeImage(exposureTimeSeconds, moveTimeSeconds);
    }

    public void takeExposure(float exposureTimeSeconds)
            throws InterruptedException, IllegalArgumentException {

        takeImage(exposureTimeSeconds, moveTimeSeconds);
    }

    /**
     * Takes a single exposure
     * @param firstBladeSetIndex index of the bladeSet to move first
     * @param exposureTimeSeconds  exposure time, on average is equal to the effective exposure time 
     * @param moveTimeSeconds time to open or close.
     * @throws InterruptedException
     * @throws IllegalArgumentException
     */
    public void takeImage(int firstBladeSetIndex, final float moveTimeSeconds, final float exposureTimeSeconds)
                throws InterruptedException, IllegalArgumentException {
        int exposureTime = (int) (1000 * exposureTimeSeconds);
        final int secondBladeSetIndex = 1 - firstBladeSetIndex;
        final float firstTargetPosition = 0;
        final float secondTargetPosition = 1;

        Timer timer1 = new Timer(exposureTime, new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                moveToPosition(secondBladeSetIndex, secondTargetPosition, moveTimeSeconds, false);
            }
        });
        timer1.setRepeats(false);

        log.info("[ShutterModule] Move bladeSets in  " + moveTimeSeconds + " seconds.");
        log.info("[ShutterModule] Taking exposure of " + exposureTimeSeconds + " seconds.");

        sendToStatus(new TakeImageStatus(firstBladeSetIndex, moveTimeSeconds, exposureTimeSeconds));
        timer1.start();
        moveToPosition(firstBladeSetIndex, firstTargetPosition, moveTimeSeconds, true);
   }

   public void takeImage(final float moveTimeSeconds, final float exposureTimeSeconds)
            throws InterruptedException, IllegalArgumentException{
        double position0 = getBladeSetPosition(0);
        double position1 = getBladeSetPosition(1);
        //System.out.println("ShutterSimulator: ***** " + position0 + " " + position1 + " *******");
        if ( ( position0 == 0 && position1 == 1) || ( position0 == 1 && position1 == 0) ) {
            final int firstBladeSetIndex = (position0 < position1) ? 1 : 0;
            takeImage(firstBladeSetIndex, moveTimeSeconds, exposureTimeSeconds);
        }  else {
            log.error("[ShutterModule] Shutter not fully closed.");
            System.out.println("ShutterModule: Shutter not fully closed. ***** ");
        }
   }

   public void closeShutter() throws InterruptedException{
        double position0 = getBladeSetPosition(0);
        double position1 = getBladeSetPosition(1);
        System.out.println("ShutterModule closeShutter: ***** " + position0 + " " + position1 + " *******");

        if ( ( position0 == 0 && position1 == 1) || ( position0 == 1 && position1 == 0) ) {
            System.out.println("Shutter already closed ***** ");
            log.info("Shutter already closed");
        }  else {
            final int firstBladeSetIndex = (position0 < position1) ? 0 : 1;
            final int secondBladeSetIndex = 1 - firstBladeSetIndex;
            final double moveTimeSeconds = MAXCLOSETIME;// * (1 - Math.max(position0, position1));
            log.info("Closing shutter");
            if (position0 == 0 || position1 == 0) {
                moveToPosition(secondBladeSetIndex, 1, (float) moveTimeSeconds);
            } else {
                System.out.println("closeShutter: moveTimeSeconds " + moveTimeSeconds + " ***** ");
                sendToStatus(new CloseShutterStatus(firstBladeSetIndex, moveTimeSeconds));
                Thread t = new Thread(new Runnable(){

                    @Override
                    public void run() {
                        
                        try {
                            Thread.sleep(100);  
                        }catch (InterruptedException ie) {
                            System.out.println(ie.getMessage());
                        }

                        moveToPosition(secondBladeSetIndex, 1 , (float) moveTimeSeconds, false);
                    }
                });
                t.start();
                moveToPosition(firstBladeSetIndex, 0, (float) moveTimeSeconds, true);
            }
        }
   }
}
