package org.lsst.ccs.subsystems.shutter.simulator.motor;

import java.util.ArrayList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

/**
 * Simulates a motor in the shutter subsystem. This is used both for 
 * simulating hardware and for predicting the motor movement in the user interface.
 * @author azemoon
 */
//FIXME: This class really implements two functions: It precomputes a set of 
// positions/accelerations and it delivers them at fixed intervals to a set of
// listeners. Perhaps these functions should be separated?
public class MotorSimulator {

    static final int PERIOD = 50;
    private ArrayList<MotorListener> listeners = new ArrayList<MotorListener>();
    //private Timer timer;
    private double velocity;
    private double acceleration;
    private double initialPosition;
    private List<MotorPosition> motorPositions = new ArrayList<MotorPosition>();
    private int index;
    private boolean retracting;

    public void addMotorListener(MotorListener l) {
        listeners.add(l);
    }

    public void removeMotorListener(MotorListener l) {
        listeners.remove(l);
    }

    /**
     * Simulate a motor movement. During the motor movement events will be delivered to
     * any motor listeners every 50ms.
     * @param targetPosition The position to which the motor should move 
     * @param moveTimeSeconds The time in seconds for the motor to move
     * @throws IllegalArgumentException if the arguments are not valid
     */
    public void simulateMovement(final float targetPosition, final double moveTimeSeconds) throws IllegalArgumentException {

        if (targetPosition < 0 || targetPosition > 1) {
            throw new IllegalArgumentException("Illegal targetPosition " + targetPosition + " outside [0,1]");
        }
        if (moveTimeSeconds <= 0) {
            throw new IllegalArgumentException("Illegal moveTimeSeconds" + moveTimeSeconds + "<=0");
        }
        retracting = targetPosition < initialPosition;
        System.out.println("===================================================");
        System.out.println("MotorSimulator: currentPosition " + initialPosition + " target position " + targetPosition);

        final int moveTime = (int) (1000 * moveTimeSeconds);
        
        getMotorPositions(targetPosition, moveTime);
        /*
        double distance = targetPosition - initialPosition;
        
        long start = System.currentTimeMillis();
        long end = start + moveTime;
        System.out.println("MotorSimulator: start " + start + " end " + end);
        final double delta = ((double) PERIOD) / moveTime;
        long time = start;
        double position = initialPosition;
        //System.out.println("MotorSimulator: delay before " + delta);
        motorPositions.add(new MotorPosition(initialPosition, 0, 0, start));
        for (double x = delta; x <= 1; x += delta) {
            //System.out.println("MotorSimulator: delay after " + delta);
            acceleration = calculateAcceleration(x);
            velocity += 4 * acceleration * delta;
            position += 2 * velocity * delta * distance;
            time += PERIOD;
            System.out.println("MotorSimulator: " + x + " " + position + " " + velocity + " " + acceleration + " " + time);
            motorPositions.add(new MotorPosition(position, velocity, acceleration, time));
        }
        motorPositions.add(new MotorPosition((double) targetPosition, 0, 0, start + moveTime));
        * 
        */
        index = 0;
        final Timer timer = new Timer();

        timer.scheduleAtFixedRate(new TimerTask() {

            //final int moveTime = (int) (1000 * moveTimeSeconds);
            //double distance = targetPosition - initialPosition;

            @Override
            public void run() {
                if (index < motorPositions.size()) {
                    //System.out.println("MotorSimulator: time " + motorPositions.get(index).getTime() + " currentTime " + System.currentTimeMillis() + " position " + motorPositions.get(index).getPosition());
                    fireMotorChangedEvent(motorPositions.get(index));
                } else {
                    fireMotorStoppedEvent();
                    initialPosition = targetPosition;
                    velocity = 0;
                    acceleration = 0;
                    timer.cancel();
                    motorPositions.clear();
                    synchronized (MotorSimulator.this) {
                        MotorSimulator.this.notifyAll();
                    }
                }
                index++;
            }
        }, 0, PERIOD);

    }
    
    private void getMotorPositions(float targetPosition, int moveTime){
        double distance = targetPosition - initialPosition;
        long start = System.currentTimeMillis();
        long end = start + moveTime;
        System.out.println("MotorSimulator: start " + start + " end " + end);
        final double delta = ((double) PERIOD) / moveTime;
        long time = start;
        double position = initialPosition;
        //System.out.println("MotorSimulator: delay before " + delta);
        motorPositions.add(new MotorPosition(initialPosition, 0, 0, start));
        for (double x = delta; x <= 1; x += delta) {
            //System.out.println("MotorSimulator: delay after " + delta);
            acceleration = calculateAcceleration(x);
            velocity += 4 * acceleration * delta;
            position += 2 * velocity * delta * distance;
            time += PERIOD;
            System.out.println("MotorSimulator: " + x + " " + position + " " + velocity + " " + acceleration + " " + time);
            motorPositions.add(new MotorPosition(position, velocity, acceleration, time));
        }
        motorPositions.add(new MotorPosition((double) targetPosition, 0, 0, start + moveTime));
        
    }
    
    private static double calculateAcceleration(double phase) {
        if (phase < .25) {
            return phase * 4;
        } else if (phase < .75) {
            return 1 - (phase - .25) * 4;
        } else {
            return phase * 4 - 4;
        }
    }

    /**
     * Gets the position that the motor will have at the beginning of the next
     * movement.
     * @return The position
     */
    public double getInitialMotorPosition() {
        return initialPosition;
    }

    public void setInitialMotorPosition(double position) {
        if (position < 0 || position > 1) {
            throw new IllegalArgumentException("Illegal targetPosition " + position + " outside [0,1]");
        }
        this.initialPosition = position;
    }

    private void fireMotorChangedEvent(MotorPosition p) {
        //TODO: This will fail if anyone adds or removes a listener
        //while this method is executing.
        for (MotorListener l : new ArrayList<MotorListener>(listeners)) {
            l.positionChanged(p);
        }
    }

    private void fireMotorStoppedEvent() {
        //TODO: This will fail if anyone adds or removes a listener
        //while this method is executing.
        for (MotorListener l : new ArrayList<MotorListener>(listeners)) {
            l.motorStopped();
        }
    }

    public boolean isRetracting() {
        return retracting;
    }

    /**
     * This method is just here to simplify testing.
     */
    synchronized void waitForMovementToComplete() throws InterruptedException {
        this.wait();
    }
}
