package org.lsst.ccs.subsystems.shutter.gui;

import data.DiscretePlotData;
import data.MetaData;
import data.MutablePlotData;
import data.PlotDataListener;
import data.SuggestedRange;
import java.util.ArrayList;
import java.util.List;
import org.freehep.util.Value;
import util.ListenerList;


/**
 *
 * @author azemoon
 */
public class HallSensorHistory implements DiscretePlotData, MutablePlotData {

    private final List<HallTransition> history = new ArrayList<HallTransition>();
    private final ListenerList<PlotDataListener> listeners = new ListenerList<PlotDataListener>();
    private final String[] names = {"Time", "Position"};
    private static double ERROR = 0.02;
    private double startTime = 0;
    private double startPosition = 0;
    private long t0 = 0;
    private double p0;

    public void HallSensorHistory() {
    }

    public void reset() {
       startTime = 0;
       t0 = 0;
       history.clear();
       firePlotDataChanged();
    }

    public void addData(long time, double position) {
        if (startTime == 0 ) {
            if ( t0 == 0) {
                t0 = time;
                p0 = position;
                return;
            } else {
                startTime = t0 - (p0 - startPosition) * ( time - t0) / (position - p0);
                history.add(new HallTransition( ( t0 - startTime)/1000.,p0, ERROR));
            }
        }
        history.add(new HallTransition( (time - startTime)/1000.,position, ERROR));
        //System.out.println("HallSensorHistory: time " + time + " startTime " + startTime + " position " + position);
        
        firePlotDataChanged();
    }

    @Override
    public int getNPoints() {
        return history.size();
    }

    @Override
    public int getNDimensions() {
        return names.length;
    }

    @Override
    public String names(int index) {
        return names[index];
    }

    @Override
    public Class types(int index) {
        return Double.TYPE;
    }

    @Override
    public void getValue(Value value, int dim, int index) {
        HallTransition transition = history.get(index);
        switch (dim) {
            case 0: value.set(transition.getTime()); break;
            case 1: value.set(transition.getPosition()); break;
            case 2: value.set(transition.getError()); break;
        }
    }

    @Override
    public MetaData getMetaData() {
        return null;
    }

    @Override
    public String getTitle() {
        return "Hall Sensor History";
    }

    @Override
    public SuggestedRange getSuggestedRange() {
        return null;
    }

    @Override
    public void addPlotDataListener(PlotDataListener listener) {
        listeners.addListener(listener);
    }

    @Override
    public void removePlotDataListener(PlotDataListener listener) {
        listeners.removeListener(listener);
    }

    private void firePlotDataChanged() {
        if (!listeners.isEmpty()) {
            for (PlotDataListener l : listeners.getListeners()) {
                l.dataChanged();
            }
        }
    }

    @Override
    public Object lock() {
        return history;
    }

    public double getStartTime() {
        return startTime;

    }

    public void setStartTime (double startTime){
        this.startTime = startTime;
    }

    public void setStartPosition (double startPosition) {
        this.startPosition = startPosition;
    }

    private static class HallTransition {

        private double time;
        private double position;
        private double error;

        public HallTransition(double time, double position, double error) {
            this.time = time;
            this.position = position;
            this.error = error;
        }

        public double getTime() {
            return time;
        }

        public double getPosition() {
            return position;
        }

        public double getError() {
            return error;
        }
    }
}
