package org.lsst.ccs.subsystem.shutter.brakewatch;

import java.util.List;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import static org.lsst.ccs.bus.states.AlertState.NOMINAL;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.alert.AlertEvent;
import static org.lsst.ccs.services.alert.AlertEvent.AlertEventType.AGENT_CONNECTION;
import static org.lsst.ccs.services.alert.AlertEvent.AlertEventType.AGENT_DISCONNECTION;
import static org.lsst.ccs.services.alert.AlertEvent.AlertEventType.ALERT_CLEARED;
import static org.lsst.ccs.services.alert.AlertEvent.AlertEventType.ALERT_RAISED;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.power.constants.PowerAgentProperties;

/**
 * Adapts the code in {@code org.lsst.ccs.subsystem.shutter.BrakePowerMonitor} in order to simply
 * log CCS bus events of interest rather than take action on any of them.
 *
 * @author tether
 */
public class BrakeWatch  extends Subsystem
        implements
        HasLifecycle,
        StatusMessageListener,
        AlertListener,
        AgentPresenceListener
{
    private static final Logger LOG = Logger.getLogger(BrakeWatch.class.getName());
    
    public BrakeWatch() {
        super("brakewatch", AgentInfo.AgentType.WORKER);
        brakeVoltageKey = "PDU_24VD/Shtr_Brakes_V";
        powerOnThreshold = 23.0;
        subsys = this;
    }
    
    private volatile AgentMessagingLayer aml;
    
    private volatile Subsystem subsys;
    
    @ConfigurationParameter(description = "The key of the brake voltage value in the KeyValueDataList.",
        isFinal = true, units="unitless")
    private volatile String brakeVoltageKey;
    
    @ConfigurationParameter(
        description = "Brake power voltage must be at least this to be considered 'on'.",
        isFinal = true, units = "V")
    private volatile double powerOnThreshold;
    
    ////////// Implementation of HasLifeCycle //////////
    /**
     * Starts listening for alerts from the target subsystem.
     */
    @Override
    public void init() {
        subsys.getAgentService(AlertService.class).addListener(this);
        subsys.getAgentService(AlertService.class).startStatusAlertListening((AgentInfo a) -> true);
        LOG.finer(() -> "Started listening for the alert " + brakeVoltageKey);
    }
    
    /**
     * Starts listening for power status messages from the publisher.
     */
    @Override
    public void postStart() {
        aml = subsys.getMessagingAccess();
        aml.addStatusMessageListener(
            this,     // Call our onStatusMessage() method.
            msg ->
                msg.getOriginAgentInfo().hasAgentProperty(PowerAgentProperties.QUAD_BOX_AGENT)
                && msg instanceof StatusSubsystemData
                && ((StatusSubsystemData)msg).getObject() instanceof KeyValueDataList
        );
        LOG.info(() -> "Started listening for brake-voltage status messages.");
        aml.getAgentPresenceManager().addAgentPresenceListener(this);
        LOG.info(() -> "Started listening for agent-presence events.");
        final List<AgentInfo> onlineAgents = aml.getAgentPresenceManager().listConnectedAgents();
        onlineAgents.stream()
                .filter( (AgentInfo agent) -> agent.hasAgentProperty(PowerAgentProperties.QUAD_BOX_AGENT))
                .forEach( (AgentInfo agent) -> {
                    LOG.log(
                            Level.INFO,
                            "Agent {0} has property {1} = {2}.",
                            new Object[]{
                                agent.toString(),  // Agent name and type.
                                PowerAgentProperties.QUAD_BOX_AGENT,
                                agent.getAgentProperty(PowerAgentProperties.QUAD_BOX_AGENT)
                            }
                    );
                });
    }
    
    /**
     * Stops listening for power status messages.
     */
    @Override
    public void postShutdown() {
        aml.removeStatusMessageListener(this);
        LOG.info(() -> "Stopped listening for status bus messages from quadbox.");
        aml.getAgentPresenceManager().removeAgentPresenceListener(this);
        LOG.info(() -> "Stopped listening for agent-presence events for quadbox.");
        subsys.getAgentService(AlertService.class).removeListener(this);
        LOG.info(() -> "Stopped listening for alerts from quadbox.");
    }
    
    ////////// Implementation of StatusMessageListener //////////

    @Override
    public void onStatusMessage(final StatusMessage rawMsg) {
        final List<KeyValueData> kvdList =
            ((KeyValueDataList)((StatusSubsystemData)rawMsg).getObject()).getListOfKeyValueData();
        Optional<KeyValueData> kvd =
            kvdList.stream()
                .filter(k -> k.getKey().equals(brakeVoltageKey))
                .findFirst();
        if (kvd.isPresent()) {
            LOG.info(() -> "Received this brake-power message: " + rawMsg.toString());
            final double voltage = (Double)kvd.get().getValue();
            sendPowerEvent( voltage >= powerOnThreshold );
        }
    }
    
    ////////// Implementation of AlertListener //////////
        
    @Override
    public void onAlert(final AlertEvent event) {
        switch (event.getType()) {
            case AGENT_CONNECTION:
                break;
            case AGENT_DISCONNECTION:
                break;
            case ALERT_CLEARED:
                {
                    // Many alerts may get cleared all at once.
                    final List<String> clearedIds = event.getClearedIds();
                    final boolean brakeAlertCleared = clearedIds.contains(brakeVoltageKey);
                    if (brakeAlertCleared) {
                        LOG.info(() -> "The brake voltage alert was cleared by agent " + event.getSource() + ".");
                        sendPowerEvent(true);
                    }
                }
                break;
            case ALERT_RAISED:
                {
                    final Alert alrt = event.getAlert();
                    if (alrt.getAlertId().equals(brakeVoltageKey)) {
                        LOG.info(() -> "The brake voltage alert was raised by agent " + event.getSource() + ".");
                        sendPowerEvent(event.getLevel() == NOMINAL);
                    }
                }
                break;
        }
    }

    ////////// Implementation of AgentPresenceListener //////////
    ////////// All methods have default implementations that do nothing.

    @Override
    public void disconnected(AgentInfo... agents) {
        // Assumes that at most one quadbox subsystem exists.
        final Optional<AgentInfo> quadBoxDisc = Stream.of(agents)
                .filter(a -> a.hasAgentProperty(PowerAgentProperties.QUAD_BOX_AGENT))
                .findFirst();
        if (quadBoxDisc.isPresent()) {
            LOG.info(() -> "The quadbox agent " + quadBoxDisc.get().toString() + " has gone off-line.");
            sendPowerEvent(false);
        }
    }
    
    // We don't test for the quadbox subsystem coming back online because that isn't enough,
    // we need to see the power status message from it.
    

    
    ////////// Event-sending code //////////
    private void sendPowerEvent(final boolean powerIsOn) {
        LOG.info(() -> "Queueing a power event with powerIsOn=" + powerIsOn);
    }

}
