package org.lsst.ccs.subsystem.shutter.statemachine;

import java.util.logging.Logger;
import static org.lsst.ccs.subsystem.shutter.statemachine.PromptReply.ACCEPTED;

/**
 * The top-level context for the state machine. Thread-safe.
 * @author tether
 */
public final class TopContext extends SimpleContext<TopContext> {

    private static final Logger LOG = Logger.getLogger(TopContext.class.getName());

    // PROTECTED by the instance lock
    private State<TopContext> synchronizingState;
    private State<TopContext> syncErrorState;
    private State<TopContext> inSyncState;
    private boolean plcEnableFlag;
    private boolean brakePowerState;
    private String syncAlertReason;
    // END PROTECTED

    /**
     * Saves the action implementations for this context and initializes the PLC-enable and brake-power
     * flags. Both flags are set to {@code false}. Also saves a default value for the
     * "contact lost" message.
     * @param actions The action implementations.
     * @throws NullPointerException if the actions ref is null.
     */
    public TopContext(final Actions actions) {
        super(actions);
        plcEnableFlag = false;
        brakePowerState = false;
        syncAlertReason = "Reason for sync error was not set.";
    }

    /**
     * Sets the flag that the {@code InSync} state will use to determine which of its
     * substates will become its current state when {@code InSync} is entered.
     * @param flag true for {@code Enabled}, false for {@code Disabled}.
     * @see Disabled
     * @see Enabled
     * @see #getPLCEnabled() 
     */
    synchronized void setPLCEnabled(final boolean flag) {plcEnableFlag = flag;}

    /**
     * Gets the flag set by {@code setPLCEnabled}.
     * @return The flag value last set.
     * @see #setPLCEnabled(boolean) 
     */
    synchronized boolean getPLCEnabled() {return plcEnableFlag;}

    /**
     * Instantiates the {@code Synchronizing} state if need be, thereafter returning
     * that instance.
     * @return The {@code Synchronizing} state object.
     */
    synchronized State<TopContext> getSynchronizingState() {
        if (synchronizingState == null) synchronizingState = new Synchronizing(this);
        return synchronizingState;
    }

    /**
     * Instantiates the {@code SyncError} state if need be, thereafter returning
     * that instance.
     * @return The {@code SyncError} state object.
     */
    synchronized State<TopContext> getSyncErrorState() {
        if (syncErrorState == null) syncErrorState = new SyncError(this);
        return syncErrorState;
    }

    /**
     * Instantiates the {@code InSync} state if need be, thereafter returning
     * that instance.
     * @return The {@code InSync} state object.
     */
    synchronized State<TopContext> getInSyncState() {
        if (inSyncState == null) inSyncState = new InSync(getActions(), this);
        return inSyncState;
    }

    /** @implNote Sets the current state to {@code Synchronizing}. */
    @Override
    public void init() {
        makeTransition(getSynchronizingState(), null);
    }

    /**
     * {@inheritDoc} Saves the most recently discovered brake power state.
     */
    @Override
    synchronized
        public void brakePowerChange(final Channel<EventReply> chan, final boolean powerIsOn)
            throws InterruptedException
    {
        LOG.fine( () -> String.format("{EVENT brakePowerChange(powerIsOn=%s) IN TopContext}", powerIsOn));
        chan.write(ACCEPTED);
        final boolean oldPowerState = brakePowerState;
        brakePowerState = powerIsOn;
        if (oldPowerState != powerIsOn) {
            // Publish the change in power state.
            getActions().publishBrakePowerStatus(brakePowerState);
        }
        if (oldPowerState && !brakePowerState) {
            // We've just lost brake power, inform the state machine.
            // We'll take a short cut by making the event a continuation of the current event
            // instead of submitting a request through the StateMachine instance's input channel.
            // We don't care what the reply to the event is.
            sendBrakePowerLossEvent();
        }
    }

    // This method exists only to be spied upon by Mockito in unit tests and must not be called from
    // other classes. Can't be declared private because then Mockito wouldn't see it.
    void sendBrakePowerLossEvent() throws InterruptedException {
        final Channel<EventReply> dummyChan = new BlackHoleChannel<>();
        this.getState().brakePowerLoss(dummyChan);
    }

    /**
     * Returns the current state of the 24V dirty power for the brakes.
     * @return True if and only if the power is on.
     */
    synchronized public boolean brakePowerIsOn() {return brakePowerState;}

    /**
     * Returns the string that will be used in the "reason" field of the SYNC alert
     * that's raised when the SyncError state is entered.
     * @return The message string that was last saved..
     */
    synchronized public String getSyncAlertReason() {
        return syncAlertReason;
    }

    /**
     * Sets the string that will be used in the "reason" field of the SYNC alert
     * that's raised when the SyncError state is entered.
     * @param reason The message string to save.
     */
    synchronized public void setSyncAlertReason(final String reason) {
        this.syncAlertReason = reason;
    }
}
