package org.lsst.ccs.subsystem.shutter;

import java.nio.ByteBuffer;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.ads.ADSDriver;
import org.lsst.ccs.drivers.ads.Notification;
import org.lsst.ccs.drivers.ads.VariableHandle;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import static org.lsst.ccs.services.alert.AlertService.RaiseAlertStrategy.ALWAYS;
import static org.lsst.ccs.services.alert.AlertService.RaiseAlertStrategy.ON_SEVERITY_CHANGE;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.shutter.PLCVariableDictionary.InVariable;
import org.lsst.ccs.subsystem.shutter.PLCVariableDictionary.OutVariable;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import static org.lsst.ccs.subsystem.shutter.common.Axis.AXIS0;
import static org.lsst.ccs.subsystem.shutter.common.Axis.AXIS1;
import org.lsst.ccs.subsystem.shutter.common.HallTransition;
import org.lsst.ccs.subsystem.shutter.common.RTD;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.MINUSX;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.PLUSX;
import org.lsst.ccs.subsystem.shutter.plc.BladeSetPosition;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.Calibrate;
import org.lsst.ccs.subsystem.shutter.plc.ChangeAxisEnablePLC;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.ClearAllFaultsPLC;
import org.lsst.ccs.subsystem.shutter.plc.ClearAxisFaultsPLC;
import org.lsst.ccs.subsystem.shutter.plc.CloseShutter;
import org.lsst.ccs.subsystem.shutter.plc.Disable;
import org.lsst.ccs.subsystem.shutter.plc.DisableAllAxesPLC;
import org.lsst.ccs.subsystem.shutter.plc.Enable;
import org.lsst.ccs.subsystem.shutter.plc.EnableAllAxesPLC;
import org.lsst.ccs.subsystem.shutter.plc.Error;
import org.lsst.ccs.subsystem.shutter.plc.GoToProd;
import org.lsst.ccs.subsystem.shutter.plc.HomeAxisPLC;
import org.lsst.ccs.subsystem.shutter.plc.Ignored;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import org.lsst.ccs.subsystem.shutter.plc.MoveAxisAbsolutePLC;
import org.lsst.ccs.subsystem.shutter.plc.MoveAxisRelativePLC;
import org.lsst.ccs.subsystem.shutter.plc.MsgToCCS;
import org.lsst.ccs.subsystem.shutter.plc.MsgToPLC;
import org.lsst.ccs.subsystem.shutter.plc.OpenShutter;
import org.lsst.ccs.subsystem.shutter.plc.PLCMsg;
import org.lsst.ccs.subsystem.shutter.plc.Reset;
import org.lsst.ccs.subsystem.shutter.plc.ShutterStatusPLC;
import org.lsst.ccs.subsystem.shutter.plc.TakeExposure;
import org.lsst.ccs.subsystem.shutter.plc.Timer;
import org.lsst.ccs.subsystem.shutter.plc.ToggleSafetyCheck;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.fromDcTime64;
import org.lsst.ccs.subsystem.shutter.sim.CubicSCurve;
import org.lsst.ccs.subsystem.shutter.sim.MotionProfile;
import org.lsst.ccs.subsystem.shutter.statemachine.Channel;
import org.lsst.ccs.subsystem.shutter.statemachine.EventReply;
import org.lsst.ccs.subsystem.shutter.statemachine.SynchronousChannel;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.subsystem.shutter.status.PtpDeviceState;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus.ShutterAxisStatus;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;
import org.lsst.ccs.utilities.scheduler.Scheduler;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * The subsystem component responsible for all communication with the Beckhoff PLC controller running the
 shutter, which includes caching the last plcStatus message(s) received from the PLC.
 *
 * @author tether
 *
 */
public class Controller implements HasLifecycle {

    @ConfigurationParameter(
        isFinal=true,
        description="IP address of the shutter controller.",
        units="unitless"
    )
    private volatile String plcIPaddr = "0.0.0.0";

    @ConfigurationParameter(
        isFinal=true,
        description="AMS address of the shutter controller.",
        units="unitless"
    )
    private volatile String plcAMSaddr = "0.0.0.0.1.1";

    @ConfigurationParameter(
        isFinal=true,
        description="Local AMS address used by the subsystem.",
        units="unitless"
    )
    private volatile String localAMSaddr = "0.0.0.1.1.1";

    @ConfigurationParameter(description="Startup delay for the notification reader.", units="s")
    private volatile Duration noticeReaderStartupDelay = Duration.ofSeconds(2);

    @ConfigurationParameter(description="Restart delay for the notification reader.", units="s")
    private volatile Duration noticeReaderRestartDelay = Duration.ofMillis(10);

    @ConfigurationParameter(description="Ack timeout for commands sent to the shutter controller.", units="s")
    private volatile Duration plcAckTimeout = Duration.ofSeconds(30);

    @ConfigurationParameter(description="The minimum exposure time.", units="s", range="PT1S..PT60S")
    private volatile Duration minExposureTime = Duration.ofMillis(1000);

    @ConfigurationParameter(isFinal=true, description="Reference positions for the -X blade set.",
            maxLength=4, units="mm")
    private volatile Map<String, Double> referenceMinusXpositions = new HashMap<>();

    @ConfigurationParameter(isFinal=true, description="Reference positions for the +X blade set.",
            maxLength=4, units="mm")
    private volatile Map<String, Double> referencePlusXpositions = new HashMap<>();

    @ConfigurationParameter(description="Move blade sets at this speed when centering.",
            units="mm/s")
    private volatile double centeringSpeed = 20.0;

    @ConfigurationParameter(description="Max speed (to attain during a stroke.", units="mm/s")
    private volatile double maxStrokeSpeed = 1667.0;

    @ConfigurationParameter(isFinal=true,
        description="Max allowed difference of a Hall transition position from prediction.", units="mm")
    private volatile double maxHallPositionError = 5.0;
    
    @ConfigurationParameter(description="Safe internal temperature range for motor controller modules.",
        units="Celsius", maxLength=2)
    private volatile Map<ShutterSide, List<Double>> safeCtrlTempRange;
    
    @ConfigurationParameter(description="Safe temperature ranges for RTD sites.",
        units="Celsius", maxLength=3)
    private volatile Map<RTD, List<Double>> safeRtdTempRange;

    private static final java.util.logging.Logger LOG = java.util.logging.Logger.getLogger(Controller.class.getName());

    // GUARDED by the instance lock.
    private ADSDriver driver;
    private PeriodicTask plcReceptionTask;
    private Map<Class<? extends MsgToPLC>, VariableHandle> writableVarHandles;
    private CountDownLatch plcReceptionStopLatch;
    private List<Axis> centeringOrder;
    private SimulatedShutter simShutter;
    // END GUARDED

    // GUARDED by the constructor.
    private final PLCVariableDictionary plcVarDict;
    private final MessageWithState pendingMessage;
    private final Scheduler eventSched;
    // END GUARDED

    @LookupField(strategy = LookupField.Strategy.TREE)
    private volatile Publisher publish;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private volatile StateMachine centralSmComponent;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private volatile Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private volatile Watchdog wdog;

    
    public Controller() {
        this.referencePlusXpositions.put(BladeSetPosition.HOME.getKey(), 0.0);
        this.referencePlusXpositions.put(BladeSetPosition.RETRACTED.getKey(), 1.0);
        this.referencePlusXpositions.put(BladeSetPosition.CENTERED.getKey(), 374.5);
        this.referencePlusXpositions.put(BladeSetPosition.EXTENDED.getKey(), 749.0);
        this.referenceMinusXpositions.put(BladeSetPosition.HOME.getKey(), 750.0);
        this.referenceMinusXpositions.put(BladeSetPosition.RETRACTED.getKey(), 749.0);
        this.referenceMinusXpositions.put(BladeSetPosition.CENTERED.getKey(), 375.5);
        this.referenceMinusXpositions.put(BladeSetPosition.EXTENDED.getKey(), 1.0);

        // See the comments for submitEvent() for why this scheduler
        // must be single-threaded.
        this.eventSched = new Scheduler("Controller tasks", 1);

        final PLCVariableDictionary dict = new PLCVariableDictionary();
        // Create dictionary entries for variables used to send messages to the PLC task.
        dict.addMsgToPLC(Calibrate.class, Calibrate::new);
        dict.addMsgToPLC(ChangeAxisEnablePLC.class, ChangeAxisEnablePLC::new);
        dict.addMsgToPLC(ChangeBrakeState.class, ChangeBrakeState::new);
        dict.addMsgToPLC(ClearAllFaultsPLC.class, ClearAllFaultsPLC::new);
        dict.addMsgToPLC(ClearAxisFaultsPLC.class, ClearAxisFaultsPLC::new);
        dict.addMsgToPLC(CloseShutter.class, CloseShutter::new);
        dict.addMsgToPLC(DisableAllAxesPLC.class, DisableAllAxesPLC::new);
        dict.addMsgToPLC(EnableAllAxesPLC.class, EnableAllAxesPLC::new);
        dict.addMsgToPLC(GoToProd.class, GoToProd::new);
        dict.addMsgToPLC(MoveAxisAbsolutePLC.class, MoveAxisAbsolutePLC::new);
        dict.addMsgToPLC(MoveAxisRelativePLC.class, MoveAxisRelativePLC::new);
        dict.addMsgToPLC(HomeAxisPLC.class, HomeAxisPLC::new);
        dict.addMsgToPLC(OpenShutter.class, OpenShutter::new);
        dict.addMsgToPLC(Reset.class, Reset::new);
        dict.addMsgToPLC(TakeExposure.class, TakeExposure::new);
        dict.addMsgToPLC(ToggleSafetyCheck.class, ToggleSafetyCheck::new);

        // Now for the variables used to send non-ack messages to CCS.
        dict.addMsgToCCS(CalibDone.class,
            CalibDone::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> centralSmComponent.calibDone(chan, (CalibDone)msg)
        );
        dict.addMsgToCCS(Disable.class,
            Disable::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> centralSmComponent.disable(chan)
        );
        dict.addMsgToCCS(Enable.class,
            Enable::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> centralSmComponent.enable(chan)
        );
        dict.addMsgToCCS(Error.class,
            Error::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> centralSmComponent.error(chan, (Error)msg)
        );
        dict.addMsgToCCS(Ignored.class,
            Ignored::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> centralSmComponent.ignored(chan, ((Ignored)msg).getReason())
        );
        dict.addMsgToCCS(MotionDonePLC.class,
            MotionDonePLC::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> centralSmComponent.motionDone(chan, (MotionDonePLC)msg)
        );
        dict.addMsgToCCS(ShutterStatusPLC.class,
            ShutterStatusPLC::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> {}  // Not an event message.
        );
        dict.addMsgToCCS(Timer.class,
            Timer::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> centralSmComponent.timer(chan)
        );
        this.plcVarDict = dict;
        this.pendingMessage = new MessageWithState();
     }

    @Override
    // Register the ShutterStatus message for trending.
    public void init() {
        final Map<ShutterSide, ShutterStatus.ShutterAxisStatus> axes = new HashMap<>();
        axes.put(PLUSX, new ShutterStatus.ShutterAxisStatus(0.0, 0.0, 0.0, false, false, false, false,
        false, 0, 0.0, false));
        axes.put(MINUSX, axes.get(PLUSX));
        final Map<RTD, Double> rtdTemp = new TreeMap<>();
        final Map<RTD, Boolean> rtdSafe = new TreeMap<>();
        for (final RTD rtd: RTD.values()) {
            rtdTemp.put(rtd, 0.0);
            rtdSafe.put(rtd, false);
        }
        // Register the top-level message of class ShutterStatus.
        final ShutterStatus exampleStatus =
            new ShutterStatus(
                0, false, 0, axes, false, rtdTemp, rtdSafe, false,
                CCSTimeStamp.currentTime(), PtpDeviceState.DISABLED, 0, false);
        subsys
            .getAgentService(DataProviderDictionaryService.class)
            .registerData(new KeyValueData("ShutterStatus", exampleStatus));
    }

    @Override
    public void shutdown() {
        terminateContact();
        eventSched.shutdown();
    }

    @Override
    public void postShutdown() {
        eventSched.shutdownNow();
    }

    /**
     * Determines if an exposure time is too small to be valid.
     * @param exposureTime The exposure time to be tested.
     * @return false if the exposure time is OK, else true.
     */
    boolean isBadExposureTime(final Duration exposureTime) {
        return exposureTime.compareTo(minExposureTime) < 0;
    }

    /**
     * Stops the notice-reception task and closes the connection to the shutter controller.
     */
    synchronized void terminateContact() {
        if (plcReceptionTask != null) {
            plcReceptionTask.cancel(true);
        }
        if (driver != null) {
            driver.close();
        }
    }

    /**
     * Makes a new connection to the shutter controller and starts the notice-reception task, but enables only
 the sending of the reset command and the reception of only the reset ack and the shutter plcStatus
 message.
     * @return true if the operation succeeded, else false.
     */
    synchronized boolean makePartialContact() {
        try {
            // 1. Stop any running notification reception task.
            if (plcReceptionTask != null) {
                plcReceptionTask.cancel(true);
            }
            // 2. Create a driver if needed.
            if (driver == null) {
                driver = new ADSDriver(localAMSaddr);
            } // 3. Close any open connection. This invalidates all extant handles
            // and cancels all notifications in effect.
            else {
                driver.close();
            }
            // 4. Forget any variable handles we've been keeping.
            if (writableVarHandles != null) {
                writableVarHandles.clear();
            }
            else {
                writableVarHandles = new HashMap<>();
            }
            // 5. Open a new connection.
            driver.open(plcAMSaddr, plcIPaddr);
            // 6. Save the variable handle for the reset() message.
            final InVariable vreset = plcVarDict.getInVariable(Reset.class);
            this.writableVarHandles.put(vreset.klass, driver.getVariableHandle(vreset.varName));
            // 7. Set up notification for reset ack variable and for the
            // plcStatus message variable.
            driver.requestNotifications(
                    driver.getVariableHandle(vreset.ackName),
                    Duration.ZERO,   // Send immediately.
                    Duration.ZERO,   // Check every PLC task cycle.
                    true);           // Send only if value has changed.
            driver.requestNotifications(driver.getVariableHandle(plcVarDict.getOutVariable(ShutterStatusPLC.class).varName),
                    Duration.ZERO, Duration.ZERO, true);
            // 8. Start the task that reads notificaions.
            subsys.getScheduler().setLogger(LOG);
            startPLCReceptionTask();
            return true;
        } catch (Exception exc) {
            LOG.log(Level.SEVERE, "Exception while trying to contact the shutter PLC.", exc);
            submitContactLost("Could not contact the shutter PLC.");
            return false;
        }
    }

    /**
     * Enable the transmission or reception, as appropriate, for all messages.
     * @return true if the operation succeeded, else false.
     */
    synchronized boolean makeFullContact() {
        try {
            // 1. Stop the processing of received messages.
            plcReceptionTask.cancel(true /* Interrupt it. */);
            plcReceptionStopLatch.await();
            // The task MUST have stopped before we proceed since the making
            // of new notification requests will generate messages that we don't want processed.
            // 2. Store handles for all "in" variables.
            final List<InVariable> inVars = plcVarDict.getAllInVariables();
            for (final InVariable var: inVars) {
                writableVarHandles.put(var.klass, driver.getVariableHandle(var.varName));
            }
            // 3. Set up notification for all "ack" variables.
            for (final InVariable var: inVars)  {
                driver.requestNotifications(
                    driver.getVariableHandle(var.ackName),
                    Duration.ZERO, Duration.ZERO, true);
            }
            // 4. Set up notification for all "out" variables.
            final List<OutVariable> outVars = plcVarDict.getAllOutVariables();
            for (final OutVariable var: outVars) {
                driver.requestNotifications(
                    driver.getVariableHandle(var.varName),
                    Duration.ZERO, Duration.ZERO, true);
            }
            // 5. Re-start the processing of received messages.
            startPLCReceptionTask();
            return true;
        }
        catch (Exception exc) {
            LOG.log(Level.SEVERE, "Exception while trying to contact the shutter PLC.", exc);
            submitContactLost("Could not contact the shutter PLC.");
            return false;
        }
    }

    /**
     * Determines whether the shutter is ready to go from Main mode to Prod mode. The determination
     * is based on the latest {@code ShutterStatusPLC} message received from the shutter PLC.
     * For the full set of conditions see {@code ShutterInfo::shutterIsReady()}.
     * @return true if all the conditions are met, else false.
     * @see ShutterInfo#shutterIsReady() 
     */
    synchronized String shutterIsReady() {
        final ShutterStatus status = publish.getShutterStatus();
        if (status == null) {
            LOG.warning("No shutter status has yet been received.");
            return "No shutter status has yet been received.";
        }
        final ShutterInfo info = new ShutterInfo(status, referenceMinusXpositions, referencePlusXpositions);
        return info.shutterIsReady(); // Also does logging.
    }

    /**
     * Used to submit a {@code contactLost()} event to the central state machine.
     * @see #submitEvent(java.lang.String, org.lsst.ccs.subsystem.shutter.PLCEventSubmitter, org.lsst.ccs.subsystem.shutter.plc.MsgToCCS)
     */
    private void submitContactLost(final String lossMsg) {
        // The msg argument of the submitter is null since this isn't an event signalled by the PLC.
        final PLCEventSubmitter submitter = (Channel<EventReply> chan, MsgToCCS msg) -> {
            centralSmComponent.contactLost(chan, lossMsg);
        };
        submitEvent("contactLost", submitter, null, Level.SEVERE);
    }

    /**
     * Submits an event to the central state machine, without blocking the thread
     * doing the submitting, by handing over the actual call of the required event
     * method and the waiting for a reply to a task created for that purpose.
     * 
     * The task is created in a single-threaded scheduler. It must be single-threaded because
     * events must be delivered in chronological order. The scheduler can always create
     * a new task quickly and if the thread is busy the task will be queued for later execution.

     * @param eventName the name of the event to be submitted, used in log messages. May not be null.
     * Do not include trailing "()", they will be added in the log message.
     * @param submitter an object whose submit() method calls the required event method and does nothing
     * else, including waiting for the reply.
     * @param msg the PLC message to be used as the argument of the event method. May be null if there
     * is no such argument.
     * @param logLevel the logging level to use when logging the rejection of the event by
     * the state machine. For example, Level.FINER if rejection isn't important or Level.WARNING
     * if it is.
     */
    private void submitEvent(
            final String eventName,
            final PLCEventSubmitter submitter,
            final MsgToCCS msg,
            final Level logLevel
    )
    {
        // First create the submission code.
        final Runnable subCode = () -> {
            try {
                final Channel<EventReply> chan = new SynchronousChannel<>();
                submitter.submit(chan, msg);
                final EventReply reply = chan.read();
                if (!reply.wasAccepted(null)) {
                    LOG.log(logLevel, "Event {0}() was rejected by the central state machine.", eventName);
                    LOG.log(logLevel, reply.getMessage(), new Object[]{});
                }
            }
            catch (InterruptedException exc) {
                LOG.warning(String.format("The submission of event %s() to the central state machine was interrupted.", eventName));
            }
            catch (TimeoutException exc) {
                // Should not be possible.
                LOG.severe(String.format("A wait for a central state machine reply to event %s() timed out.", eventName));
            }
        };
        // Run the submission code.
        eventSched.schedule(subCode, 0, TimeUnit.MILLISECONDS);
    }

    /**
     * Determines whether the shutter is ready to perform a calibration. The determination
     * is based on the latest {@code ShutterStatusPLC} message received from the shutter PLC. 
     * For the full list of conditions see {@code ShutterInfo::readyForCalibration()}.
     * @return true if all the conditions are met, else false.
     * @see ShutterInfo#readyForCalibration() 
     */
    synchronized boolean readyForCalibration() {
        final ShutterStatus status = publish.getShutterStatus();
        if (status == null) {
            LOG.warning("No shutter status has yet been received.");
            return false;
        }
        final ShutterInfo info = new ShutterInfo(status, referenceMinusXpositions, referencePlusXpositions);
        return info.readyForCalibration();  // Also does logging.
    }
    /**
     * Decides in which order to move the blade sets, most extended first, then starts
     * the first motion.
     */
    synchronized void startFirstCentering() {
        final ShutterStatus status = publish.getShutterStatus();
        final ShutterInfo info = new ShutterInfo(status, referenceMinusXpositions, referencePlusXpositions);
        assert status != null : "Null shutter status";
        assert centeringOrder == null : "A centering order should not exist yet.";
        centeringOrder = info.getCenteringOrder();
        centralSmComponent.getActions().relay(
            new MoveAxisAbsolutePLC(
                new MoveAxisAbsolute(
                    centeringOrder.get(0).getName(),
                    info.getCentered(centeringOrder.get(0)),
                    centeringSpeed
                )
            )
        );
    }

    /**
     * Starts the second and last blade set centering motion. {@code startFirstCentering()}
     * should have been called first.
     */
    synchronized void startSecondCentering() {
        assert centeringOrder != null : "A centering order wasn't chosen.";
        final ShutterStatus status = publish.getShutterStatus();
        final ShutterInfo info = new ShutterInfo(status, referenceMinusXpositions, referencePlusXpositions);
        centralSmComponent.getActions().relay(
            new MoveAxisAbsolutePLC(
                new MoveAxisAbsolute(
                    centeringOrder.get(1).getName(),
                    info.getCentered(centeringOrder.get(1)),
                    centeringSpeed
                )
            )
        );
        centeringOrder = null;
    }


    private void startPLCReceptionTask() throws InterruptedException {
        final ADSDriver drv = driver;
        final CountDownLatch startLatch = new CountDownLatch(1);
        final CountDownLatch stopLatch = new CountDownLatch(1);
        final Runnable reception = () ->
            plcReceptionTaskBody(drv, plcVarDict, pendingMessage, publish, centralSmComponent, startLatch, stopLatch);
        plcReceptionStopLatch = stopLatch;
        plcReceptionTask
                = subsys
                .getScheduler()
                .scheduleWithFixedDelay(reception,
                        noticeReaderStartupDelay.toMillis(),
                        noticeReaderRestartDelay.toMillis(),
                        TimeUnit.MILLISECONDS,
                        "Notification reception",
                        Level.SEVERE);
        startLatch.await();
    }

    public void checkHallTransitions(final MotionDonePLC motplc) {
        // Construct the predicted trajectory calculator.
        final MotionDone mot = motplc.getStatusBusMessage();
        final MotionProfile profile =
            new CubicSCurve(
                mot.endPosition() - mot.startPosition(),
                1e-3 * mot.actualDuration().toMillis());
        final Instant tstart = mot.startTime().getTAIInstant();
        int badCount = 0;

        try {  // Don't let bad Hall data stop state transitions. JIRA issue LSSTCCSSHUTTER-150.
            // For each Hall transition htran...
            for (final HallTransition htran: mot.hallTransitions()) {
                // Get the elapsed time t between the transition and the start of motion.
                final double t =
                    1e-3 * Duration.between(tstart, htran.getTime().getTAIInstant()).toMillis();
                // Calculate the predicted position p at time t.
                final double p =  mot.startPosition() + profile.distance(t);
                // Get the actual transition position a.
                final double a = htran.getPosition();
               // Test for too great a deviation.
               if (Math.abs(p - a) > maxHallPositionError) {
                   ++badCount;
                   LOG.warning(
                       String.format(
                           "Hall ID %d transition at %g mm, prediction = %g mm.",
                           htran.getSensorId(),
                           a,
                           p
                       ));
               }
            }
        }
        catch (Exception exc) {
            LOG.log(Level.WARNING, "An exception was thrown during Hall data analysis.", exc);
            Alerts.MOTION.raise(
                subsys.getAgentService(AlertService.class),
                AlertState.WARNING,
                String.format(
                    "%s was thrown during Hall data analysis. Check the log file.",
                    exc.getClass().getSimpleName()),
                ALWAYS);
        }

        // Alert if we had big deviations.
        if (badCount > 0) {
            Alerts.MOTION.raise(
                subsys.getAgentService(AlertService.class),
                AlertState.WARNING,
                String.format(
                    "There were %d Hall transitions more than %g mm off prediction.",
                    badCount, maxHallPositionError),
                ALWAYS);
        }
    }

    ////////// PLC reception tasks //////////

    // PLC task state numbers appearing in the plcStatus messages from the controller.
    private static final int PLC_DISABLED_STATE = 10000;
    private static final int PLC_STILL_STATE = 21010;

    private void plcReceptionTaskBody
    (
        final ADSDriver driver,
        final PLCVariableDictionary plcVarDict,
        final MessageWithState pendingMessage,
        final Publisher publish,
        final StateMachine machine,
        final CountDownLatch startLatch,
        final CountDownLatch stopLatch
    )
    {
        // Clear the notice queue.
        final List<Notification> leftovers = new ArrayList<>();
        driver.drainNotifications(leftovers);
        // Allow the main thread to proceed.
        startLatch.countDown();
        // Start reading notices from the queue.
        LOG.info("The notice-reading task has started.");
        try {
            while (true) {
                // Take a new notice from the queue. Terminate loop if interrupted.
                final Notification notice = driver.takeNotification();
                // Take note that we've received a PLC message.
                wdog.countMessage();
                // Get the name of the variable and the message data from the notice.
                final String varName = notice.getVariableHandle().getName();
                final ByteBuffer data = notice.getData();
                processMessageData(varName, data, plcVarDict, pendingMessage, publish, machine);
            }
        }
        catch (InterruptedException exc) {
            stopLatch.countDown();
            LOG.info("Normal stop of the notice-reading loop.");
        }
    }

    private void simReceptionTaskBody
    (
        final PLCVariableDictionary plcVarDict,
        final MessageWithState pendingMessage,
        final Publisher publish,
        final StateMachine machine,
        final BlockingQueue<SimMessage> simulationQueue
    )
    {
        // Start reading notices from the queue.
        LOG.info("The simulated notice-reading loop has started.");
        try {
            while (true) {
                // Take a new notice from the queue. Terminate loop if interrupted.
                final SimMessage msg = simulationQueue.take();
                // Take note that we've received a PLC message.
                wdog.countMessage();
                // Get the name of the variable and the message data from the notice.
                final String varName = msg.varName;
                final ByteBuffer data = msg.data;
                LOG.info("Read message with var name " + varName);
                processMessageData(varName, data, plcVarDict, pendingMessage, publish, machine);
            }
        }
        catch (InterruptedException exc) {
            LOG.info("Normal stop of the simulated notice-reading loop.");
        }
    }

    private void processMessageData(
        final String varName,
        final ByteBuffer data,
        final PLCVariableDictionary plcVarDict,
        final MessageWithState pendingMessage,
        final Publisher publish,
        final StateMachine machine) throws InterruptedException
    {
        // Check that the message-set-version is present and correct.
        // If it isn't, signal loss of contact and skip the rest of the message processing.
        if (checkMessageSetVersion(data) ) {return;}
        // Perform lookups in the dictionary of PLC variables.?
        final InVariable ackVar = plcVarDict.getInVariable(varName);
        final OutVariable outVar = plcVarDict.getOutVariable(varName);
        assert (ackVar != null) || (outVar != null): "Unknown message type received from PLC.";
        assert (ackVar == null) || (outVar == null): "Ambiguous message received from PLC.";
        // If it's an acknowledegment, handle the ack.
        if (ackVar != null) {
            LOG.fine("Calling handleAckMessage().");
            handleAckMessage(data, ackVar, pendingMessage);
            LOG.fine("Returned from handleAckMessage()");
        }
        // If it's a plcStatus message. handle that.
        else if (outVar != null && outVar.klass == ShutterStatusPLC.class) {
            LOG.fine("Calling handleStatusMessage().");
            handleStatusMessage(data, outVar, publish, machine);
            LOG.fine("Returned from handleStatusMessage().");
        }
        // Otherwise it must be an event message, so submit it to the state centralSmComponent.
        else if (outVar != null) {
            LOG.fine("Calling handleEventMessage");
            handleEventMessage(data, outVar);
            LOG.fine("Returned from handleEventMessage().");
        }
    }

    private boolean checkMessageSetVersion(
        final ByteBuffer data
        )
    {
        boolean failure = false;
        if (PLCMsg.messageVersionIsBad(data)) { // Will issue a log message if needed.
            submitContactLost("Wrong message-set version in message from PLC.");
            failure = true;
        }
        return failure;
    }

    private void handleAckMessage(
        final ByteBuffer data,
        final InVariable ackVar,
        final MessageWithState pending
    )
    {
        MsgToPLC msg;
        try {
            msg = ackVar.ackDecoder.apply(data);
            final MessageWithState.Disposition disp = pending.ack(msg);
            switch (disp) {
                case OK:
                case WRONG_ACK:
                    // We were expecting an ack and we got one. The caller of awaitAck()
                    // will handle an incorrect ack.
                    break;
                case WRONG_STATE:
                    LOG.severe(
                        String.format(
                            "Received an unexpected ack message of type %s.",
                            msg.getClass().getSimpleName()));
                    submitContactLost("Unexpected command ack from PLC.");
                    break;
                default:
                    LOG.warning(String.format("Unexpected status of %s from ack processing.", disp));
            }
        }
        catch(Exception exc) {
            LOG.log(Level.SEVERE, "Error during ack processing.", exc);
        }
    }

    // Updates the saved ShutterStatus message and publishes it on the CCS status bus. Also submits a
    // plcIsDisabled() event to the central state machine if the PLC state is Disabled or submits
    // a plcISEnabled() event if the PLC state is Still.
    private void handleStatusMessage(
        final ByteBuffer data,
        final OutVariable outVar,
        final Publisher publish,
        final StateMachine machine) throws InterruptedException
    {

        ShutterStatusPLC plcStatus;
        try {
            plcStatus = (ShutterStatusPLC)outVar.decoder.apply(data);
        }
        catch (Exception exc) {
            LOG.log(Level.SEVERE, "Error during status message decoding.", exc);
            return;
        }
        LOG.fine("Received ShutterStatusPLC. " + plcStatus.toString());
        final Map<ShutterSide, ShutterAxisStatus> axes = new EnumMap<>(ShutterSide.class);
        for (final ShutterSide side: ShutterSide.values()) {
            final ShutterStatusPLC.AxisStatusPLC axplc = plcStatus.getAxisStatus(side);
            final ShutterAxisStatus ax = new ShutterAxisStatus(
                axplc.getActPos(),
                axplc.getActVel(),
                axplc.getSetAcc(),
                axplc.isEnabled(),
                axplc.isBrakeEngaged(),
                axplc.atLowLimit(),
                axplc.atHighLimit(),
                axplc.isHomed(),
                axplc.getErrorID(),
                axplc.getCtrlTemp(),
                axplc.getCtrlTemp() >= safeCtrlTempRange.get(side).get(0) 
                    && axplc.getCtrlTemp() <= safeCtrlTempRange.get(side).get(1)
            );
            if (!ax.hasSafeTemp()) {
                final String msg = String.format(
                    "%s controller temperature of %7.1f \u2103 is outside of the range [%7.1f, %7.1f].",
                        side,
                        ax.getCtrlTemp(),
                        safeCtrlTempRange.get(side).get(0),
                        safeCtrlTempRange.get(side).get(1));
                Alerts.MOTOR.raise(
                    subsys.getAgentService(AlertService.class),
                    AlertState.ALARM,
                    msg,
                    ON_SEVERITY_CHANGE);
            }
            axes.put(side, ax);
        }
        
        final Map<RTD, Double> temperature = new TreeMap<>();
        final Map<RTD, Boolean> tempIsSafe = new TreeMap<>();
        for (final RTD rtd: RTD.values()) {
            final double t = RTD.rawToCelsius(plcStatus.getTemperature().get(rtd.getIndex()));
            temperature.put(rtd, t);
            tempIsSafe.put(
                rtd,
                t >= safeRtdTempRange.get(rtd).get(0) && t <= safeRtdTempRange.get(rtd).get(1));
        }
        for (final RTD rtd: RTD.values()) {
            if (!tempIsSafe.get(rtd)) {
                final String msg = String.format(
                    "%s temperature of %7.1f \u2103 is outside of the range [%7.1f, %7.1f].",
                    rtd,
                    temperature.get(rtd),
                    safeRtdTempRange.get(rtd).get(0),
                    safeRtdTempRange.get(rtd).get(1));
                Alerts.MOTOR.raise(
                    subsys.getAgentService(AlertService.class),
                    AlertState.ALARM,
                    msg,
                    ON_SEVERITY_CHANGE);
            }
        }
        publish.updateShutterStatus(
            new ShutterStatus(
                plcStatus.getMotionProfile(),
                plcStatus.isCalibrated(),
                plcStatus.getSmState(),
                axes,
                plcStatus.isSafetyOn(),
                temperature,
                tempIsSafe,
                machine.brakePowerIsOn(),
                fromDcTime64(plcStatus.getRawCreationTime(), plcStatus.getLeapSeconds()),
                PtpDeviceState.fromStateNumber(plcStatus.getPtpState()),
                plcStatus.getLeapSeconds(),
                plcStatus.isLeapValid()
            )
        );
        if (plcStatus.getSmState() == PLC_DISABLED_STATE) {
            final PLCEventSubmitter submitter = (replyChan, msg) -> {machine.plcIsDisabled(replyChan);};
            submitEvent("plcIsDisabled", submitter, null, Level.FINER);
        }
        else if (plcStatus.getSmState() == PLC_STILL_STATE) {
            final PLCEventSubmitter submitter = (replyChan, msg) -> {machine.plcIsEnabled(replyChan);};
            submitEvent("plcIsEnabled", submitter, null, Level.FINER);
        }
    }

    // Handle a message from the PLC that directly corresponds to a central state machine event.
    // Use the decoder and event submitter that were registered for the message in the PLC variable dictionary.
    private void handleEventMessage(final ByteBuffer data, final OutVariable outVar)
    throws InterruptedException
    {
        MsgToCCS msg;
        try {
            msg = outVar.decoder.apply(data);
        }
        catch (Exception exc) {
            LOG.log(Level.SEVERE, 
                    String.format(
                            "Error during the decoding of event message %s from the PLC.", 
                            outVar.varName
                    ),
                    exc
            );
            return;
        }
        // Submit the event to the central state machine. For the event name use the message class
        // name converted to the corresponding method name.
        final String clsnm = msg.getClass().getSimpleName();
        final String eventName = clsnm.substring(0, 1).toLowerCase() + clsnm.substring(1);
        submitEvent(eventName, outVar.submitter, msg, Level.SEVERE);
     }

    ////////// End of PLC reception tasks //////////

    synchronized void resetPLC() {
        relay(new Reset(maxStrokeSpeed));
    }

    void simulateResetPLC() {
        simulateRelay(new Reset(maxStrokeSpeed));
    }

    void relay(final MsgToPLC msg) {
        LOG.fine(String.format("Sending message %s to PLC.", msg.getClass().getSimpleName()));
        final MessageWithState.Disposition disp = sendPendingMessage(msg);
        switch (disp) {
            case OK:
                finishPendingMessage();
                break;
            case SEND_ERROR:
                submitContactLost("Unable to send a message to the PLC.");
                break;
            default:
                LOG.warning(String.format("Unexpected message sending disposition of %s.", disp));
                break;
        }
    }

    private MessageWithState.Disposition sendPendingMessage(final MsgToPLC msg) {
        final InVariable inVar = plcVarDict.getInVariable(msg.getClass());
        assert inVar != null: "Message not recognized by relay().";
        final VariableHandle handle = writableVarHandles.get(msg.getClass());
        assert handle != null: "No handle for writing the PLC message variable.";
        return pendingMessage.send(driver, msg, handle);
    }

    void simulateRelay(final MsgToPLC msg) {
        LOG.info(String.format("Sending message %s to simulated PLC.", msg.getClass().getSimpleName()));
        final MessageWithState.Disposition disp = pendingMessage.simSend(simShutter, msg);
        switch (disp) {
            case OK:
                finishPendingMessage();
                break;
            case SEND_ERROR:
                submitContactLost("Unable to send a message to the PLC.");
                break;
            default:
                LOG.warning(String.format("Unexpected message sending disposition of %s.", disp));
                break;
        }
    }

    private BlockingQueue<SimMessage> simulationQueue;

    synchronized void simulateContact() {
        if (plcReceptionTask == null) {
            final Runnable reception = () -> {
                simReceptionTaskBody(plcVarDict, pendingMessage, publish, centralSmComponent, simulationQueue);
            };
            simulationQueue = new ArrayBlockingQueue<>(1000);
            plcReceptionTask
                    = subsys
                            .getScheduler()
                            .scheduleWithFixedDelay(reception,
                                    noticeReaderStartupDelay.toMillis(),
                                    noticeReaderRestartDelay.toMillis(),
                                    TimeUnit.MILLISECONDS,
                                    "Simulated notification reception",
                                    Level.SEVERE);
            simShutter = new SimulatedShutter(subsys.getScheduler(), simulationQueue, plcVarDict);
            simShutter.start();
        }

    }

    private void finishPendingMessage() {
        LOG.fine(String.format("Waiting %s for ack.", plcAckTimeout));
        MessageWithState.Disposition disp;
        try {
            disp = pendingMessage.awaitAck(plcAckTimeout);
        }
        catch (InterruptedException exc) {
            LOG.warning("Interrupted while waiting for an ack from the shutter controller.");
            return;
        }
        LOG.fine("Done waiting.");
        switch (disp) {
            case OK:
                LOG.fine("Correct ack received from the shutter controller.");
                break;
            case WRONG_STATE:
                LOG.severe("Attempted to wait for an ack from the shutter controller when none was expected.");
                break;
            case WRONG_ACK:
                LOG.severe("Ack from the shutter controller had the wrong message ID.");
                submitContactLost("Command ack from the PLC had wrong command ID.");
                break;
            case ACK_TIMEOUT:
                LOG.severe("Timed out waiting for an ack from the shutter controller.");
                submitContactLost("Timed out waiting for a command ack from the PLC.");
                break;
            default:
                LOG.severe(String.format("Unexpected status of %s from awaitAck().", disp));
                break;
        }
    }

    private static class MessageWithState {
        // Thread-safe.
        // One thread calls send() to atomically transmit a message and change the state
        // to MESSAGE_SENT (if all went well). After exiting send() it then calls waitForAck()
        // provided that the send() call worked. A second thread calls ack() to atomically submit
        // an ack message for comparison to the sent message, if any.

        public static enum State {NO_MESSAGE, MESSAGE_SENT, MESSAGE_ACKED;}
        public static enum Disposition {OK, SEND_ERROR, WRONG_ACK, ACK_TIMEOUT, WRONG_STATE;}

        // GUARDED by the constructor.
        private final BlockingQueue<Disposition> result;
        // END GUARDED

        // GUARDED by instance lock
        private MsgToPLC sentMsg;
        private State curstate;
        // END GUARDED

        public MessageWithState() {
            this.sentMsg = null;
            this.result = new ArrayBlockingQueue<>(1);
            this.curstate = State.NO_MESSAGE;
        }

        /**
         * Atomically performs the following:
         * <ol>
         * <li>If the state is not NO_MESSAGE then return WRONG_STATE.</li>
         * <li>Attempt to send a message to the PLC.</li>
         * <li>If the attempt failed then return SEND_ERROR.</li>
         * <li>If the attempt succeeded then change state to MESSAGE_SENT,
         *     save the message and clear the result queue for awaitAck()
         *     and return OK.</li>
         * </ol>
         * @param driver The ADS driver.
         * @param msg The message to send.
         * @param handle The handle of the PLC variable to which the message is to be written.
         * @return A Disposition value.
         */
        public synchronized Disposition send(
            final ADSDriver driver,
            final MsgToPLC msg,
            final VariableHandle handle
        )
        {
            Disposition disp = Disposition.OK;
            if (curstate == State.NO_MESSAGE) {
                try {
                    final ByteBuffer buf = handle.createBuffer();
                    msg.encode(buf);
                    driver.writeVariable(handle, buf);
                    curstate = State.MESSAGE_SENT;
                    sentMsg = msg;
                    result.poll(); // Make sure queue is empty.
                } catch (DriverException exc) {
                    LOG.log(Level.SEVERE, "Sending of a message to the shutter PLC failed.", exc);
                    disp = Disposition.SEND_ERROR;
                }
            } else {
                // Not in state NO_MESSAGE.
                disp = Disposition.WRONG_STATE;
            }
            return disp;
        }

        /**
         * Like send() but for use in simulation mode.
         * @param msg
         * @return A disposition value.
         * @see #send(org.lsst.ccs.drivers.ads.ADSDriver, org.lsst.ccs.subsystem.shutter.plc.MsgToPLC, org.lsst.ccs.drivers.ads.VariableHandle)
         */
        public synchronized Disposition simSend(
            final SimulatedShutter sim,
            final MsgToPLC msg
        )
        {
            Disposition disp = Disposition.OK;
            if (curstate == State.NO_MESSAGE) {
                try {
                    sim.accept(msg);
                    curstate = State.MESSAGE_SENT;
                    sentMsg = msg;
                    result.poll(); // Make sure queue is empty.
                } catch (Exception exc) {
                    LOG.log(Level.SEVERE, "Sending of a message to the shutter simulation failed.", exc);
                    disp = Disposition.SEND_ERROR;
                }
            } else {
                // Not in state NO_MESSAGE.
                disp = Disposition.WRONG_STATE;
            }
            return disp;
        }

        /**
         * Atomically performs the following:
         * <ol>
         * <li>If the state is not MESSAGE_SENT then return WRONG_STATE.</li>
         * <li>If the ack doesn't match the sent message:
         *     <ol type=a>
         *     <li>Set state to MESSAGE_ACKED.</li>
         *     <li>Put WRONG_ACK in the result queue.</li>
         *     <li>Return WRONG_ACK.</li>
         *     </ol>
         * </li>
         * <li> if the ack does match then:
         *     <ol type=a>
         *     <li>Set state to MESSAGE_ACKED.</li>
         *     <li> Put OK in the result queue.</li>
         *     <li>Return OK.</li>
         *     </ol>
         * </li>
         * </ol>
         *
         * @param ackMsg The acknowledgment message.
         * @return A Disposition value.
         */
        public synchronized Disposition ack(final MsgToPLC ackMsg) throws InterruptedException {
            if (curstate != State.MESSAGE_SENT) {
                return Disposition.WRONG_STATE;
            }
            curstate = State.MESSAGE_ACKED;
            if (sentMsg.hasSameIdent(ackMsg)) {
                result.put(Disposition.OK);
                return Disposition.OK;
            }
            result.put(Disposition.WRONG_ACK);
            return Disposition.WRONG_ACK;
        }

        /**
         * Waits for a message disposition sent by the ack() method. When this method returns
         * the state will be NO_MESSAGE once again and the result queue will be empty.
         * @param timeout How long to wait.
         * @return A disposition value.
         */
        public Disposition awaitAck(final Duration timeout) throws InterruptedException {
            try {
                final Disposition disp = result.poll(timeout.toMillis(), TimeUnit.MILLISECONDS);
                return (disp != null) ? disp : Disposition.ACK_TIMEOUT;
            }
            finally {
                synchronized(this) {curstate = State.NO_MESSAGE; result.poll();}
            }
        }


    }

    // Used to answer questions about the shutter based on the latest plcStatus report and
    // some configuration parameters.
    static class ShutterInfo {

        private final ShutterStatus status;
        private final Map<Axis, Map<BladeSetPosition, Double>> config;

        ShutterInfo(
            final ShutterStatus status,
            final Map<String, Double> minusXConfig,
            final Map<String, Double> plusXConfig)
        {
            this.status = status;
            this.config = new EnumMap<>(Axis.class);
            final Map<Axis, Map<String, Double>> conf = new EnumMap<>(Axis.class);
            conf.put(Axis.getPlusXSide(), plusXConfig);
            conf.put(Axis.getMinusXSide(), minusXConfig);
            for (final Axis ax: Axis.values()) {
                final Map<BladeSetPosition, Double> posmap = new EnumMap<>(BladeSetPosition.class);
                for (final BladeSetPosition pos: BladeSetPosition.values()) {
                    posmap.put(pos, conf.get(ax).get(pos.getKey()));
                }
                this.config.put(ax, posmap);
            }
        }

        private double getParam(final Axis axis, final BladeSetPosition pos) {return config.get(axis).get(pos);}
        public double getHome(final Axis axis) {return getParam(axis, BladeSetPosition.HOME);}
        public double getRetracted(final Axis axis) {return getParam(axis, BladeSetPosition.RETRACTED);}
        public double getCentered(final Axis axis) {return getParam(axis, BladeSetPosition.CENTERED);}
        public double getExtended(final Axis axis) {return getParam(axis, BladeSetPosition.EXTENDED);}
        public boolean isRetracted(final Axis axis) {
            final double pos = status.getAxisStatus(ShutterSide.fromAxis(axis)).getActPos();
            return Math.abs(pos - getHome(axis)) <= Math.abs(getRetracted(axis) - getHome(axis));
        }
        public boolean isExtended(final Axis axis) {
            final double pos = status.getAxisStatus(ShutterSide.fromAxis(axis)).getActPos();
            return Math.abs(pos - getHome(axis)) >= Math.abs(getExtended(axis) - getHome(axis));
        }
        public boolean shutterIsClosed() {
            return (isRetracted(AXIS0) && isExtended(AXIS1))
                   ||
                   (isExtended(AXIS0) && isRetracted(AXIS1));
        }
        public boolean axesAreEnabled() {
            return status.getAxisStatus(PLUSX).isEnabled() && status.getAxisStatus(MINUSX).isEnabled();
        }
        public boolean brakesAreReleased() {
            return !status.getAxisStatus(PLUSX).isBrakeEngaged()
                && !status.getAxisStatus(MINUSX).isBrakeEngaged()
                && status.brakePowerIsOn();
        }
        public boolean axesAreHomed() {
            return status.getAxisStatus(PLUSX).isHomed() && status.getAxisStatus(MINUSX).isHomed();
        }
        public boolean shutterIsCalibrated() {return status.isCalibrated();}
        public boolean isSafetyOn() {return status.isSafetyOn();}
        public String shutterIsReady() {
            boolean ready = true;
            final StringBuilder errors = new StringBuilder("The shutter is not ready.");
            if (!shutterIsClosed()) {
                ready = false;
                errors.append("\nIt's not completely closed.");
            }
            if (!axesAreEnabled()) {
                ready = false;
                errors.append("\nAt least one axis is disabled.");
            }
            if (!brakesAreReleased()) {
                ready = false;
                errors.append("At least one brake is still engaged.");
            }
            if (!axesAreHomed()) {
                ready = false;
                errors.append("\nAt least one axis needs to be homed.");
            }
            if (!shutterIsCalibrated()) {
                ready = false;
                errors.append("\nIt's not calibrated.");
            }
            if (!isSafetyOn()) {
                ready = false;
                errors.append("\nSafety checks are not enabled.");
            }
            if (!ready) {
                LOG.warning(errors.toString());
            }
            return ready ? null : errors.toString();
        }

        public boolean readyForCalibration() {
            boolean ready = true;
            if (!axesAreEnabled()) {
                LOG.warning("Shutter not ready - axes not enabled.");
                ready = false;
            }
            if (!brakesAreReleased()) {
                LOG.warning("Shutter not ready - brakes not released.");
                ready = false;
            }
            return ready;
        }
        public List<Axis> getCenteringOrder() {
            final double extent1 =
                Math.abs(status.getAxisStatus(ShutterSide.fromAxis(AXIS0)).getActPos() - getHome(AXIS0));
            final double extent2 =
                Math.abs(status.getAxisStatus(ShutterSide.fromAxis(AXIS1)).getActPos() - getHome(AXIS1));
            if (extent1 >= extent2) {
                return Arrays.asList(AXIS0, AXIS1);
            }
            else {
                return Arrays.asList(AXIS1, AXIS0);
            }
        }
    }
}
