package org.lsst.ccs.subsystem.shutter.statemachine;

import java.time.Duration;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.HomeAxis;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.Calibrate;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.Error;
import org.lsst.ccs.subsystem.shutter.plc.Ignored;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;

/**
 * Owner of a set of states and manager of transitions between them. Has a name,
 * a current state and a set of action implementations.
 * Is responsible for calling state entry methods, transition actions and
 * state exit methods in the correct order. Contexts may occur at many levels in
 * a state machine. The main context contains the state machine as a whole. A composite
 * state is a context for its substates and is a state for an owning context.
 * <p>
 * A context implements methods for all events, handling some itself and forwarding
 * the rest to the current state. This interface defines default event methods which
 * all forward to the current state.
 * <p>
 * A context provides an initialization method which makes sure that the right state
 * is the current state, if need be creating that state and making an internal transition
 * to it. This method is called whenever the context is entered. That's just once for the top-level
 * context but for a composite state it's whenever a transition occurs to it.
 * <p>
 * A context creates each state that it manages at most once. Normally this will happen on
 * the first transition to a state.
 *
 * @param <C> The concrete subclass that's extending this one. This allows state
 * fields defined here to given the type {@code State<C>} rather than just {@code State}.
 * @see CompositeState
 * @author tether
 */
interface Context<C extends Context<?>> extends Events {

    /**
     * Gets the name of this context.
     * @return the name string.
     */
    String getName();

    /**
     * Gets the current state of this context.
     * @return The current state.
     */
    State<C> getState();

    /**
     * Prepares the context for execution.
     * @implSpec At a minimum any implementation must make sure that the right state is current.
     */
    void init();

    /**
     * Cleans up a context that's not in active use. For example, resets the state to null.
     */
    void finish();

    /**
     * Obtains the set of action implementations owned by this context.
     * @return
     */
    Actions getActions();

    /**
     * Performs a state transition, calling the appropriate exit and
     * entry methods while running the transition action in between.
     * @param newState the next value for the current state.
     * @param action The action, possibly null, associated with the transition.
     * @throws NullPointerException if {@code newState} is null.
     */
    void makeTransition(State<C> newState, Runnable action);

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void brakePowerChange(final Channel<EventReply> chan, final boolean isPowerOn) throws InterruptedException {
        getState().brakePowerChange(chan, isPowerOn);
    }
    
    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void brakePowerLoss(final Channel<EventReply> chan) throws InterruptedException {
        getState().brakePowerLoss(chan);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void contactLost(final Channel<EventReply> chan, String msg) throws InterruptedException {
        getState().contactLost(chan, msg);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void ignored(final Channel<EventReply> chan, final Ignored.Reason reason) throws InterruptedException {
        getState().ignored(chan, reason);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void homeAxis(final Channel<EventReply> chan, final HomeAxis req) throws InterruptedException {
        getState().homeAxis(chan, req);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void plcIsEnabled(final Channel<EventReply> chan) throws InterruptedException {
        getState().plcIsEnabled(chan);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void plcIsDisabled(final Channel<EventReply> chan) throws InterruptedException {
        getState().plcIsDisabled(chan);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void resync(final Channel<EventReply> chan) throws InterruptedException {
        getState().resync(chan);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void syncTimeout(final Channel<EventReply> chan) throws InterruptedException {
        getState().syncTimeout(chan);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void enable(final Channel<EventReply> chan) throws InterruptedException {
        getState().enable(chan);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void disable(final Channel<EventReply> chan) throws InterruptedException {
        getState().disable(chan);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void motionDone(final Channel<EventReply> chan, MotionDonePLC mot) throws InterruptedException {
        getState().motionDone(chan, mot);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void calibrate(final Channel<EventReply> chan, Calibrate cal) throws InterruptedException {
        getState().calibrate(chan, cal);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void toggleSafetyCheck(final Channel<EventReply> chan) throws InterruptedException {
        getState().toggleSafetyCheck(chan);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void calibDone(final Channel<EventReply> chan, CalibDone cal) throws InterruptedException {
        getState().calibDone(chan, cal);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void error(final Channel<EventReply> chan, final Error err) throws InterruptedException {
        getState().error(chan, err);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void reset(final Channel<EventReply> chan) throws InterruptedException {
        getState().reset(chan);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void takeExposure(final Channel<EventReply> chan, Duration exposureTime) throws InterruptedException {
        getState().takeExposure(chan, exposureTime);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void openShutter(final Channel<EventReply> chan) throws InterruptedException {
        getState().openShutter(chan);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void timer(final Channel<EventReply> chan) throws InterruptedException {
        getState().timer(chan);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void closeShutter(final Channel<EventReply> chan) throws InterruptedException {
        getState().closeShutter(chan);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void gotoProd(final Channel<EventReply> chan) throws InterruptedException {
        getState().gotoProd(chan);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default void gotoCenter(final Channel<EventReply> chan) throws InterruptedException {
        getState().gotoCenter(chan);
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    default void moveAxisAbsolute(final Channel<EventReply> chan, MoveAxisAbsolute req) throws InterruptedException {
        getState().moveAxisAbsolute(chan, req);
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    default void moveAxisRelative(final Channel<EventReply> chan, MoveAxisRelative req) throws InterruptedException {
        getState().moveAxisRelative(chan, req);
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    default void clearAllFaults(final Channel<EventReply> chan, ClearAllFaults req) throws InterruptedException {
        getState().clearAllFaults(chan, req);
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    default void changeAxisEnable(final Channel<EventReply> chan, ChangeAxisEnable req) throws InterruptedException {
        getState().changeAxisEnable(chan, req);
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    public default void changeBrakeState(final Channel<EventReply> chan, Axis ax, ChangeBrakeState.State newState) throws InterruptedException {
        getState().changeBrakeState(chan, ax, newState);
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    default void clearAxisFaults(final Channel<EventReply> chan, ClearAxisFaults req) throws InterruptedException {
        getState().clearAxisFaults(chan, req);
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    default void enableAllAxes(final Channel<EventReply> chan, EnableAllAxes req) throws InterruptedException {
        getState().enableAllAxes(chan, req);
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    default void disableAllAxes(final Channel<EventReply> chan, DisableAllAxes req) throws InterruptedException {
        getState().disableAllAxes(chan, req);
    }
}
