package org.lsst.ccs.subsystem.shutter.statemachine;

import java.time.Duration;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.HomeAxis;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.Calibrate;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.Error;
import org.lsst.ccs.subsystem.shutter.plc.Ignored;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;

/**
 * The set of all events that are recognized by the state machine used by the CCS subsystem
 * to control the camera shutter. An event may cause actions to be performed and may also
 * trigger a state transition. An void instance is sent by each event method indicating
 * acceptance or rejection. Events are the inputs to the state machine.
 * @see void
 * @author tether
 */
public interface Events {

    /**
     * Indicates that the 24V dirty power for the brakes has been lost or regained.
     * @param chan The channel used to deliver the event reply.
     * @param powerIsOn True if and only if the power is known to be on. Otherwise the power
     * is either known to be off or not known to be on or off.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void brakePowerChange(Channel<EventReply> chan, boolean powerIsOn) throws InterruptedException;
    
    /**
     * Indicates that we've just lost power to the brakes. The default behavior is to ignore this
     * event except for states in which motion is taking place.
     * @param chan The channel used to deliver the event reply.
     * @throws InterruptedException 
     */
    void brakePowerLoss(Channel<EventReply> chan) throws InterruptedException;

    /**
     * Indicates that the subsystem can't communicate properly with the shutter PLC.
     * @param chan The channel used to deliver the event reply.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
   void contactLost(Channel<EventReply> chan) throws InterruptedException;

    /** Indicates that the shutter controller has sent a status message in which reported
     *  state is Disabled. This event is ignored save when synchronizing the subsystem
     *  state and the PLC state. At other times we rely on explicit {@code enable()}
     *  and {@code disable()} event messages sent from the controller.
     * @param chan The channel used to deliver the event reply.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void plcIsEnabled(Channel<EventReply> chan) throws InterruptedException;

    /** Indicates that shutter controller has sent a status message in which the reported state
     *  is Enabled. At other times we rely on explicit {@code enable()}
     *  and {@code disable()} event messages sent from the controller.
     * @param chan The channel used to deliver the event reply.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void plcIsDisabled(Channel<EventReply> chan) throws InterruptedException;

    /** Triggers another attempt to synchronize states with the PLC.
     * @param chan The channel used to deliver the event reply.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void resync(Channel<EventReply> chan) throws InterruptedException;

    /** Indicates that after a reset the PLC has taken too long to reach one of
     * the designated after-reset states.
     * @param chan The channel used to deliver the event reply.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void syncTimeout(Channel<EventReply> chan) throws InterruptedException;

    /**Indicates that the shutter hardware protection is is allowing shutter operation.
     * @param chan The channel used to deliver the event reply.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void enable(Channel<EventReply> chan) throws InterruptedException;

    /** Indicates that the shutter hardware protection is forbidding shutter operation.
     * @param chan The channel used to deliver the event reply.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void disable(Channel<EventReply> chan) throws InterruptedException;

    /** Indicates that a blade set has successfully completed a motion.
     * @param chan The channel used to deliver the event reply.
     * @param profileData The motion profile.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void motionDone(Channel<EventReply> chan, MotionDonePLC profileData) throws InterruptedException;

    /** Requests a Hall calibration.
     * @param chan The channel used to deliver the event reply.
     * @param calibParams The calibration parameters. If null the event is rejected.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void calibrate(Channel<EventReply> chan, Calibrate calibParams) throws InterruptedException;

    /**
     * Reverses the current setting of the PLC safety check flag, true->false, false->true. Note
     * that the flag will be reset to true after any motion completes or after a general reset. Any motion
     * that takes place with the flag set to false is limited in speed to at most 1 mm/sec.
     * @param chan The channel used to deliver the event reply.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void toggleSafetyCheck(Channel<EventReply> chan) throws InterruptedException;

    /** Indicates that a Hall calibration operation has been successful.
     * @param chan The channel used to deliver the event reply.
     * @param calibResults The results of the calibration.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void calibDone(Channel<EventReply> chan, CalibDone calibResults) throws InterruptedException;

    /** Indicates that the shutter tried and failed to carry out a requested operation.
     * @param chan The channel used to deliver the event reply.
     * @param err The error message received from the shutter.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void error(Channel<EventReply> chan, Error err) throws InterruptedException;

    /** Requests that the shutter controller stop any motion in progress and
     *  reset its state machine to one of its initial states.
     * @param chan The channel used to deliver the event reply.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void reset(Channel<EventReply> chan) throws InterruptedException;

    /** Requests that an exposure of the given duration be started.
     * @param chan The channel used to deliver the event reply.
     * @param exposureTime The length of the exposure. If null or too short a duration the
     * event is rejected.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void takeExposure(Channel<EventReply> chan, Duration exposureTime) throws InterruptedException;

    /** Requests that an opening of the shutter be started. Rejected if the shutter is not closed.
     * @param chan The channel used to deliver the event reply.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void openShutter(Channel<EventReply> chan) throws InterruptedException;

    /** Indicates that an exposure in progress has run for its allotted time.
     * @param chan The channel used to deliver the event reply.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void timer(Channel<EventReply> chan) throws InterruptedException;

    /** Requests that a closing of the shutter be started. Rejected if the shutter is not open.
     * @param chan The channel used to deliver the event reply.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void closeShutter(Channel<EventReply> chan) throws InterruptedException;

    /** Indicates that the shutter PLC state machine has ignored an event sent to it.
     * @param chan The channel used to deliver the event reply.
     * @param reason The reason why the event was rejected.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void ignored(Channel<EventReply> chan, Ignored.Reason reason) throws InterruptedException;

    /** Requests that the shutter, if it's ready to do so, accept subsequent requests to take exposures.
     * @param chan The channel used to deliver the event reply.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void gotoProd(Channel<EventReply> chan) throws InterruptedException;

    /**
     * Requests that the shutter perform a homing operation for the given axis.
     * @param chan The channel used to deliver the event reply.
     * @param req Contains the request parameters. If null then the event is rejected.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void homeAxis(Channel<EventReply> chan, HomeAxis req) throws InterruptedException;

    /** Requests that the shutter prepare for power-down by placing both blade sets in the center
     * position.
     * @param chan The channel used to deliver the event reply.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void gotoCenter(Channel<EventReply> chan) throws InterruptedException;

    /** Requests that the given axis be moved to a given absolute position.
     * @param chan The channel used to deliver the event reply.
     * @param req Contains the request parameters. If null then the event is rejected.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void moveAxisAbsolute(Channel<EventReply> chan, MoveAxisAbsolute req) throws InterruptedException;

    /** Requests that an axis change its position by a certain offset.
     * @param chan The channel used to deliver the event reply.
     * @param req Contains the request parameters. If null then event is rejected.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void moveAxisRelative(Channel<EventReply> chan, MoveAxisRelative req) throws InterruptedException;

    /** Requests that fault conditions be cleared for both axes.
     * @param chan The channel used to deliver the event reply.
     * @param req Contains the request parameters. If null then the event is rejected.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void clearAllFaults(Channel<EventReply> chan, ClearAllFaults req) throws InterruptedException;

    /** Requests that the enable state of the given axis be changed.
     * @param chan The channel used to deliver the event reply.
     * @param req Contains the request parameters. If null then the event is rejected.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void changeAxisEnable(Channel<EventReply> chan, ChangeAxisEnable req) throws InterruptedException;

    /** Requests that the brake state on a given axis be changed.
     * @param chan The channel used to deliver the event reply.
     * @param ax The PLC task axis to affect.
     * @param newState The desired brake state.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void changeBrakeState(Channel<EventReply> chan, Axis ax, ChangeBrakeState.State newState) throws InterruptedException;

    /** Requests that fault conditions be cleared for the given axis.
     * @param chan The channel used to deliver the event reply.
     * @param req Contains the request parameters. If null then the event is rejected.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void clearAxisFaults(Channel<EventReply> chan, ClearAxisFaults req) throws InterruptedException;

    /** Requests that both axes be enabled.
     * @param chan The channel used to deliver the event reply.
     * @param req Contains the request parameters. If null then event is rejected.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void enableAllAxes(Channel<EventReply> chan, EnableAllAxes req) throws InterruptedException;

    /** Requests that both axes be disabled.
     * @param chan The channel used to deliver the event reply.
     * @param req Contains the request parameters. If null then the event is rejected.
     * @throws InterruptedException if inter-task communication is interrupted.
     */
    void disableAllAxes(Channel<EventReply> chan, DisableAllAxes req) throws InterruptedException;

}
