package org.lsst.ccs.subsystem.shutter.statemachine;

import java.time.Duration;
import java.util.Objects;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.HomeAxis;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.Calibrate;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.Error;
import org.lsst.ccs.subsystem.shutter.plc.Ignored;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import static org.lsst.ccs.subsystem.shutter.statemachine.PromptReply.ACCEPTED;

/**
 * A state which has no substates. Adds a context field and
 * default event method implementations which always reject, except
 * for {@code plcIsEnabled()} and {@code plcIsDisabled()} which by default are
 * silently ignored. Also adds default state names equal to the simple names of their classes.
 * @author tether
 */
abstract class SimpleState<C extends Context<?>> implements State<C> {

    private final C context;

    /**
     * Saves this state's context.
     * @param context The context.
     * @throws NullPointerException if the context is null.
     */
    protected SimpleState(final C context) {
        Objects.requireNonNull(context, "A state's context can't be null.");
        this.context = context;
    }

    @Override
    public final C getContext() {return context;}

    /**
     * @implNote This default implementation returns the simple name of this state's class.
     */
    @Override
    public String getName() {return this.getClass().getSimpleName();}


    /**
     * @implNote This default implementation always rejects the event with a message saying
     * that the event should have been already handled by the top context.
     */
    @Override
    public void brakePowerChange(final Channel<EventReply> chan, final boolean isPowerOn)
        throws InterruptedException
    {
        chan.write(new PromptReply("The brake power-change event should have been handled by the top level."));
    }
    
    /**
     * @implNote This default implementation accepts the event but does nothing.
     * @throws InterruptedException 
     */
    @Override
    public void brakePowerLoss(final Channel<EventReply> chan) throws InterruptedException {
        chan.write(ACCEPTED);
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
   public void contactLost(final Channel<EventReply> chan) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always accepts the event but does nothing.
     */
   @Override
    public void plcIsEnabled(final Channel<EventReply> chan) throws InterruptedException {
        chan.write(ACCEPTED);
    }

    /**
     * @implNote This default implementation always accepts the event but does nothing.
     */
    @Override
    public void plcIsDisabled(final Channel<EventReply> chan) throws InterruptedException {
        chan.write(ACCEPTED);
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void resync(final Channel<EventReply> chan) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void syncTimeout(final Channel<EventReply> chan) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void enable(final Channel<EventReply> chan) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void disable(final Channel<EventReply> chan) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void motionDone(final Channel<EventReply> chan, final MotionDonePLC profileData) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void toggleSafetyCheck(final Channel<EventReply> chan) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void calibrate(final Channel<EventReply> chan, final Calibrate calibParams) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void calibDone(final Channel<EventReply> chan, final CalibDone calibResults) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void error(final Channel<EventReply> chan, final Error err) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void reset(final Channel<EventReply> chan) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void takeExposure(final Channel<EventReply> chan, final Duration exposureTime) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void openShutter(final Channel<EventReply> chan) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void timer(final Channel<EventReply> chan) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void closeShutter(final Channel<EventReply> chan) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void ignored(final Channel<EventReply> chan, final Ignored.Reason reason) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void homeAxis(final Channel<EventReply> chan, final HomeAxis req) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void gotoProd(final Channel<EventReply> chan) throws InterruptedException {
        chan.write(new PromptReply("Can't go to Prod from CCS state " + getName() + "."));
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public void gotoCenter(final Channel<EventReply> chan) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public void moveAxisAbsolute(final Channel<EventReply> chan, final MoveAxisAbsolute req) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public void moveAxisRelative(final Channel<EventReply> chan, final MoveAxisRelative req) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public void clearAllFaults(final Channel<EventReply> chan, final ClearAllFaults req) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public void changeAxisEnable(final Channel<EventReply> chan, final ChangeAxisEnable req) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public void clearAxisFaults(final Channel<EventReply> chan, final ClearAxisFaults req) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public void changeBrakeState(
            final Channel<EventReply> chan,
            final Axis ax, final ChangeBrakeState.State newState
    )
    throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public void enableAllAxes(final Channel<EventReply> chan, final EnableAllAxes req) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public void disableAllAxes(final Channel<EventReply> chan, final DisableAllAxes req) throws InterruptedException {
        chan.write(new PromptReply(rejectionMsg()));
    }

    /**
     * Creates an event-rejection message naming both the state name and the event name. Must be called
     * directly from the event method in order to work properly.
     * @return The rejection message.
     */
    protected final String rejectionMsg() {
        return String.format("Event %s() is invalid for CCS state %s.",
                             new Exception().getStackTrace()[1].getMethodName(), // Caller of this method.
                             getName());
    }

}
