package org.lsst.ccs.subsystem.shutter.plc;

import java.nio.ByteBuffer;
import java.time.Duration;
import java.time.Instant;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Provides some small utility functions.
 * @author tether
 */
public class Tools {

    /**
     * Gets a PLC BOOL value from a byte buffer and converts it to a boolean value.
     * @param data Holds the data.
     * @return The boolean value.
     */
    public static boolean getBoolean(final ByteBuffer data) {return data.get() != 0;}

    /**
     * Appends to a byte buffer the PLC BOOL value equivalent to a Java boolean.
     * @param data Holds the data.
     * @param b The boolean value to convert and append.
     */
    public static void putBoolean(final ByteBuffer data, final boolean b)
    {data.put(b ? (byte)1 : (byte)0);}

    public static long UTC_SECONDS_YEAR_2000 = 946_684_800;

    /**
     * Converts a TwinCAT absolute timestamp to a CCSTimeStamp. Absolute timestamps
     * are produced by the PLC function F_ConvTcToExtTime64() and are expressed
     * as 64-bit unsigned integers of type T_DCTIME64, a count of nanoseconds.
     * <p>
     * T_DCTIME64 timestamps are like TAI64N timestamps but have a couple of 
     * peculiarities. A value of zero nanoseconds denotes 0 hours on Jan 1 2000 UTC, not
     * any of the standard epochs. Even so, the timestamp has the full, current count of leap
     * seconds even though many of those occurred before the year 2000. Therefore producing
     * a true TAI64N value must be done with care in order to avoid double-counting
     * those earlier leap seconds.
     * <p>
     * @param dctime64 The T_DCTIME64 value.
     * @param dtai What the observatory calls the the number of seconds that a 
     * "TAI Instant" is ahead of a standard Instant. It's effectively the current number
     * of leap seconds, 37 as of November 2023.
     * @return The CCS timestamp.
     */
    public static CCSTimeStamp fromDcTime64(final long dctime64, final int dtai) {
        // 
        // (1) Perform an unsigned division of the T_DCTIME64 value by one billion
        //     in order to get a count of seconds.
        // (2) Calculate the remainder to get a count of nanoseconds.
        // (3) Add to the result of step 2 the number of UTC milliseconds between the DC time epoch
        //     and the Java/Unix epoch of 0 hours on Jan 1, 1970.
        // (4) Construct an Instant from the counts of seconds and nanoseconds.
        // (5) Give the Instant and the DTAI to fromTAI().
        long seconds = Long.divideUnsigned(dctime64, 1_000_000_000L);  // (1)
        long nanos = dctime64 - 1_000_000_000L * seconds;              // (2)
        final Instant inst =
            Instant
            .ofEpochSecond(seconds, nanos)                             // (4)
            .plusSeconds(UTC_SECONDS_YEAR_2000);                       // (3)
        return CCSTimeStamp.fromTAI(inst, dtai);                       // (5)
    }
    
    /**
     * Converts a CCS timestamp to a T_DCTIME64 value. Essentially the inverse of
     * {@link #fromDcTime64(long, int)} except that the DTAI value is not supplied. It has to be
     * calculated from the CCS timestamp itself since we need the value of DTAI as it was
     * when the timestamp was created, not the current value.
     * @param ccsTime The CCS timestamp.
     * @return The absolute T_DCTIME64 timestamp, which should be treated as unsigned.
     */
    public static long toDcTime64(final CCSTimeStamp ccsTime) {
        final Instant tai = ccsTime.getTAIInstant();      // Already has DTAI added.
        long seconds = tai.minusSeconds(UTC_SECONDS_YEAR_2000).getEpochSecond();
        long nanos = tai.getNano();
        return seconds * 1_000_000_000L + nanos;
    }
    
    /**
     * Converts EtherCat time interval, assumed to be the difference of two DC time values,
     * into a {@code Duration}.
     * @param dcNanos The DC time difference, a count of nanoseconds.
     * @return The {@code Duration} value.
     */
    public static Duration fromDcDuration(final long dcNanos) {
        return Duration.ofNanos(dcNanos);
    }
    
    /**
     * Reverses the transformation performed by {@link #toDcDuration(java.time.Duration) }.
     * @param interval The time interval to convert.
     * @return The count of nanoseconds.
     */
    public static long toDcDuration(final Duration interval) {
        return interval.toNanos();
    }

    public static void main(final String[] args) {
        // Check whether the two absolute time conversions are inverses.
        final CCSTimeStamp now = CCSTimeStamp.currentTime();
        final int tdai = (int)
            Duration
                .between(now.getUTCInstant(), now.getTAIInstant())
                .getSeconds();
        System.out.println("TDAI (seconds): " + Integer.toString(tdai));
        final long dcNow = toDcTime64(now);
        final CCSTimeStamp now2 = fromDcTime64(dcNow, tdai);
        System.out.println("Original CCS timestamp: " + now.toString());
        System.out.println("Converted to T_DCTIME64: 0x" + Long.toHexString(dcNow));
        System.out.println("Converted back to CCS timestamp: " + now2.toString());
    }
}
