package org.lsst.ccs.subsystem.shutter.plc;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.getBoolean;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.putBoolean;

/**
 * Holds a ShutterStatusPLC message sent from the PLC.
 * @author tether
 */
public class ShutterStatusPLC extends MsgToCCS {

    private final int motionProfile;

    private final boolean isCalibrated;

    private final int smState;

    private final Map<ShutterSide, AxisStatusPLC> axstatus;

    private final boolean isSafetyOn;

    private final List<Integer> temperature;
    
    private final long when;
    
    private final int ptpState;
    
    private final int leapSeconds;
    
    private final boolean leapIsValid;

    /**
     * Constructs from a sequence number and field values.
     * @param sequence A message sequence number.
     * @param motionProfile the motion profile number.
     * @param isCalibrated Is the shutter calibrated?
     * @param smState the state of the PLC state machine.
     * @param axstatus the two axis status objects, one per side.
     * @param isSafetyOn Are the safety checks in effect?
     * @param temperature the RTD readout values.
     * @param when the absolute TAI time at which this message was created, in TwinCAT T_DCTIME64 format.
     * @param ptpState the state of the PTP slave on the EL6688 device.
     * @param leapSeconds the current number of leap seconds obtained from PTP packets.
     * @param leapIsValid true if and only if the leapSecond count is deemed valid by the EL6688.
     */
    public ShutterStatusPLC(
        final int sequence,
        final int motionProfile,
        final boolean isCalibrated,
        final int smState,
        final Map<ShutterSide, AxisStatusPLC> axstatus,
        final boolean isSafetyOn,
        final List<Integer> temperature,
        final long when,
        final int ptpState,
        final int leapSeconds,
        final boolean leapIsValid
    )
    {
        super(sequence);
        this.motionProfile = motionProfile;
        this.isCalibrated = isCalibrated;
        this.smState = smState;
        this.axstatus = axstatus;
        this.isSafetyOn = isSafetyOn;
        this.temperature = temperature;
        this.when = when;
        this.ptpState = ptpState;
        this.leapSeconds = leapSeconds;
        this.leapIsValid = leapIsValid;
    }

    /**
     * Reads and converts the PLC form of the message.
     * @param data Holds the PLC message data.
     */
    public ShutterStatusPLC(final ByteBuffer data) {
        super(data);
        this.motionProfile = data.getInt();
        this.isCalibrated = getBoolean(data);
        this.smState = data.getInt();
        final Map<ShutterSide, AxisStatusPLC> axes = new EnumMap<>(ShutterSide.class);
        axes.put(ShutterSide.fromAxis(Axis.AXIS0), decodeAxisStatus(data));
        axes.put(ShutterSide.fromAxis(Axis.AXIS1), decodeAxisStatus(data));
        this.axstatus = axes;
        this.isSafetyOn = getBoolean(data);
        final List<Integer> temp = new ArrayList<>(3);
        temp.add(data.getInt());
        temp.add(data.getInt());
        temp.add(data.getInt());
        this.temperature = temp;
        this.when = data.getLong();
        this.ptpState = data.getInt();
        this.leapSeconds = data.getInt();
        this.leapIsValid = getBoolean(data);
    }

    private static AxisStatusPLC decodeAxisStatus(final ByteBuffer data) {
        final double actPos = data.getDouble();
        final double actVel = data.getDouble();
        final double setAcc = data.getDouble();
        final boolean enabled = getBoolean(data);
        final boolean brakeEngaged = getBoolean(data);
        final boolean lowLimit = getBoolean(data);
        final boolean highLimit = getBoolean(data);
        final boolean isHomed = getBoolean(data);
        final int errorID = data.getInt();
        final double ctrlTemp = data.getDouble(); // May be labeled "motorTemp" in PLC code.
        return new AxisStatusPLC(actPos, actVel, setAcc, enabled, brakeEngaged,
            lowLimit, highLimit, isHomed, errorID, ctrlTemp);
    }

    @Override
    public void encode(final ByteBuffer data) {
        super.encode(data);
        data.putInt(getMotionProfile());
        putBoolean(data, isCalibrated());
        data.putInt(getSmState());
        for (final Axis ax: Axis.values()) {
            encodeAxisStatus(data, getAxisStatus(ShutterSide.fromAxis(ax)));
        }
        putBoolean(data, isSafetyOn());
        getTemperature().forEach(data::putInt);
        data.putLong(getRawCreationTime());
        data.putInt(getPtpState());
        data.putInt(getLeapSeconds());
        putBoolean(data, isLeapValid());
    }

    private static void encodeAxisStatus(final ByteBuffer data, final AxisStatusPLC status) {
        data.putDouble(status.getActPos());
        data.putDouble(status.getActVel());
        data.putDouble(status.getSetAcc());
        putBoolean(data, status.isEnabled());
        putBoolean(data, status.isBrakeEngaged());
        putBoolean(data, status.atLowLimit());
        putBoolean(data, status.atHighLimit());
        putBoolean(data, status.isHomed());
        data.putInt(status.getErrorID());
        data.putDouble(status.getCtrlTemp());
    }

    /**
     * Gets the motion profile number.
     * @return The number.
     */
    public int getMotionProfile() {
        return motionProfile;
    }

    /**
     * Is the shutter calibrated?
     * @return
     */
    public boolean isCalibrated() {
        return isCalibrated;
    }

    /**
     * Returns the state of the PLC state machine.
     * @return The state number.
     */
    public int getSmState() {
        return smState;
    }

    /**
     * Gets the {@code AxisStatusPLC} value for the given side.
     * @return The axis status object.
     */
    public AxisStatusPLC getAxisStatus(final ShutterSide side) {
        return axstatus.get(side);
    }

    /**
     * Are the safety checks on?
     * @return true if the checks are one, else false.
     */
    public boolean isSafetyOn() {
        return isSafetyOn;
    }
    
    /**
     * Gets the RTD readouts.
     * @return the three RTD values.
     */
    public List<Integer> getTemperature() {
        return new ArrayList<>(temperature);
    }
    
    /**
     * Gets the absolute TAI message creation time in TwinCAT T_DCTIME64 format.
     * @return the message creation time.
     */
    public long getRawCreationTime() {
        return when;
    }
    
    /**
     * Gets the integer PTP state value that was read from the PTP slave of the EL6688.
     * @return the state value.
     * @see org.lsst..ccs.subsystem.shutter.status.PtpSlaveState
     */
    public int getPtpState() {
        return ptpState;
    }
    
    /**
     * Get the current number of leap seconds derived from PTP packets received by the EL6688.
     * @return the number of leap seconds.
     */
    public int getLeapSeconds() {
        return leapSeconds;
    }
    
    /**
     * Is the count of leap seconds valid? It won't be if the EL6688 has received no PTP packets, for example.
     * @return the validity flag.
     */
    public boolean isLeapValid() {
        return leapIsValid;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("ShutterStatusPLC{");
        sb.append(super.toString());
        sb.append(", motionProfile=").append(motionProfile);
        sb.append(", isCalibrated=").append(isCalibrated);
        sb.append(", smState=").append(smState);
        sb.append(", axstatus=").append(axstatus);
        sb.append(", isSafetyOn=").append(isSafetyOn);
        sb.append(", temperature=").append(temperature);
        sb.append(", when=0x").append(Long.toHexString(when));
        sb.append(", ptpState=").append(ptpState);
        sb.append(", leapSeconds=").append(leapSeconds);
        sb.append(", leapIsValid=").append(leapIsValid);
        sb.append('}');
        return sb.toString();
    }


    /**
     * Holds the status report for a single axis in a {@code ShutterStatusPLC} message. Immutable.
     */
    public final static class AxisStatusPLC {
        private final double actPos;
        private final double actVel;
        private final double setAcc;
        private final boolean enabled;
        private final boolean brakeEngaged;
        private final boolean lowLimit;
        private final boolean highLimit;
        private final boolean isHomed;
        private final int errorID;
        private final double ctrlTemp;

        /** Constructs from scratch.
         *
         * @param actPos See {@link #getActPos() }
         * @param actVel See {@link #getActVel() }
         * @param setAcc See {@link #getSetAcc() }
         * @param enabled See {@link #isEnabled() }
         * @param brakeEngaged See {@link #isBrakeEngaged() }
         * @param lowLimit See {@link #atLowLimit() }
         * @param highLimit See {@link #atHighLimit() }
         * @param isHomed See {@link #isHomed() }
         * @param errorID See {@link #getErrorID() }
         * @param ctrlTemp See {@link #getCtrlTemp() }
         */
        public AxisStatusPLC(
            final double actPos,
            final double actVel,
            final double setAcc,
            final boolean enabled,
            final boolean brakeEngaged,
            final boolean lowLimit,
            final boolean highLimit,
            final boolean isHomed,
            final int errorID,
            final double ctrlTemp
        )
        {
            this.actPos = actPos;
            this.actVel = actVel;
            this.setAcc = setAcc;
            this.enabled = enabled;
            this.brakeEngaged = brakeEngaged;
            this.lowLimit = lowLimit;
            this.highLimit = highLimit;
            this.isHomed = isHomed;
            this.errorID = errorID;
            this.ctrlTemp = ctrlTemp;
        }

        /**
         * Constructs a string of the form 'AxisStatusPLC{fieldname=value, ...}'
         * @return The string.
         */
        @Override
        public String toString() {
            return "AxisStatusPLC{" + "actPos=" + actPos + ", actVel=" + actVel + ", setAcc=" + setAcc +
                   ", enabled=" + enabled + ", brakeEngaged=" + brakeEngaged + ", lowLimit=" + lowLimit +
                   ", highLimit=" + highLimit + ", isHomed=" + isHomed + ", errorID=" + errorID +
                   ", ctrlTemp=" + ctrlTemp + '}';
        }

        /**
         * Gets the actual position of the axis.
         * @return The position in mm.
         */
        public double getActPos() {
            return actPos;
        }

        /**
         * Gets the actual velocity of the axis.
         * @return The velocity in mm/sec.
         */
        public double getActVel() {
            return actVel;
        }

        /**
         * Gets the "set", or commanded, acceleration of the axis.
         * @return The acceleration in mm/sec/sec.
         */
        public double getSetAcc() {
            return setAcc;
        }

        /**
         * Is the axis enabled for motion?
         * @return true if it is, else false.
         */
        public boolean isEnabled() {
            return enabled;
        }

        /**
         * Is the brake set on this axis?
         * @return true if it is, else false.
         */
        public boolean isBrakeEngaged() {
            return brakeEngaged;
        }

        /**
         * Is the axis at the the low limit switch?
         * @return true if it is, else false.
         */
        public boolean atLowLimit() {
            return lowLimit;
        }

        /**
         * Is the axis at the high limit switch?
         * @return true if it is, else false.
         */
        public boolean atHighLimit() {
            return highLimit;
        }

        /**
         * Has the homing operation been performed on this axis since the last controller reset?
         * @return true if it has, else false.
         */
        public boolean isHomed() {
            return isHomed;
        }

        /**
         * Gets the error code for the axis.
         * @return The error code set by the Beckhoff motion control task.
         */
        public int getErrorID() {
            return errorID;
        }

        /**
         * Gets the operating temperature of the axis motor controller.
         * @return The temperature in degrees Celsius.
         */
        public double getCtrlTemp() {
            return ctrlTemp;
        }
    }

}
