package org.lsst.ccs.subsystem.shutter.plc;

import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.logging.Logger;

/**
 * The base class for all messages sent CCS and PLC in either direction.
 * @author tether
 */
public abstract class PLCMsg {

    private static final Logger LOG = Logger.getLogger(PLCMsg.class.getName());

    /**
     * The version number for the set of messages exchanged by CCS and PLC.
     */
    public static int MESSAGE_SET_VERSION = 3; // September 2023. PTP info in some messages.

    /**
     * Checks whether the message just received (or encoded) has the correct message-set version
     * number.
     * @param data The byte buffer containing the message starting at its current position.
     * @return true if the version number is present and correct, else false.
     */
    public static boolean messageVersionIsBad(final ByteBuffer data) {
        try {
            // The version should be in the next four bytes. Look at them without
            // changing the current position.
            final int ver = data.getInt(  data.position() );
            if (ver == MESSAGE_SET_VERSION) {
                return false;
            }
            else {
                LOG.severe(
                    String.format(
                        "Bad version in message received from shutter. Expected %d, got %d."
                        ,MESSAGE_SET_VERSION,
                        ver));
                return true;
            }
        }
        catch (BufferUnderflowException exc) {
            LOG.severe("Message received from shutter is too short to contain a version number.");
            return true;
        }
    }

    private final int version;

    /**
     * Sets the version number to MESSAGE_SET_VERSION.
     * @see #MESSAGE_SET_VERSION
     */
    protected PLCMsg() {
        this.version = MESSAGE_SET_VERSION;
    }

    /**
     * Reads and converts the version field of a PLC message.
     * @param data Holds the PLC message data.
     */
    protected PLCMsg(final ByteBuffer data) {
        this.version = data.getInt();
    }

    /**
     * Converts the message fields to PLC form and appends them to a byte buffer.
     * @param data Holds the PLC message data.
     */
    public void encode(final ByteBuffer data) {
        data.putInt(version);
    }

    /**
     * Gets the message-set version number embedded in this message.
     * @return The version number.
     */
    public int getVersion() {
        return version;
    }

    /**
     * Converts the message to a readable string.
     * @return The string form of the message. The short message class name
     * will come first followed by the field names and values enclosed in curly braces.
     * For example, if message Foo is sent from CCS to the PLC it will inherit
     * from MsgToPLC and look something like this: FooMessage{MsgToPLC{PLCMsg{...} ...} field1=val1, ...}
     */
    @Override
    public String toString() {
        return "PLCMsg{" + "version=" + version + '}';
    }
}
