package org.lsst.ccs.subsystem.shutter.plc;

import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.common.EncoderSample;
import org.lsst.ccs.subsystem.shutter.common.HallTransition;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.PLUSX;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.fromDcDuration;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.fromDcTime64;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.getBoolean;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.putBoolean;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.toDcDuration;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.toDcTime64;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.subsystem.shutter.status.PtpDeviceState;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Contains a motion-done message sent from the PLC. Contains an internal
 * instance of {@code MotionDone}
 * @see MotionDone
 * @author tether
 */
public class MotionDonePLC extends MsgToCCS {

    private static final Logger LOG = Logger.getLogger(MotionDonePLC.class.getName());
    
    final MotionDone motion;

    /** Constructs from scratch.
     *  @param sequence A message sequence number.
     *  @param motion A reference of {@code MotionDone} which this object will own.
     */
    public MotionDonePLC(final int sequence, final MotionDone motion) {
        super(sequence);
        this.motion = motion;
    }

    /**
     * Reads and converts the PLC form of the message.
     * @param data The PLC message data.
     */
    public MotionDonePLC(final ByteBuffer data) {
        super(data);
        final MotionDone.Builder builder =
            new MotionDone.Builder()
                .side(ShutterSide.fromAxis(Axis.fromAxisNum(data.getInt())));
        
        final long dcTime64Start = data.getLong(); // Can't convert this just yet.
        LOG.log(Level.FINER,
            "Raw motion start time: {0}", new Object[]{Long.toUnsignedString(dcTime64Start)});
        
        builder
                .startPosition(data.getDouble())
                .targetDuration(fromDcDuration(data.getLong()))
                .targetPosition(data.getDouble())
                .endPosition(data.getDouble())
                .actualDuration(fromDcDuration(data.getLong()))
                .hallTransitions(new ArrayList<>())
                .encoderSamples(new ArrayList<>());
        final int ntrans = data.getInt();
        final int maxtrans = data.getInt();
        final int nsamples = data.getInt();
        final int maxsamples = data.getInt();
        
        // We can't convert the timestamps in the Hall transitions or the
        // encoder samples without the leap second count, which is at
        // the end of the message. So mark our place and skip down
        // to the PTP-related data.
        final int tranStart = data.position();
        data.position(data.position() + maxtrans * HALL_TRANSITION_SIZE + maxsamples * ENCODER_SAMPLE_SIZE);
        builder
            .ptpState(PtpDeviceState.fromStateNumber(data.getInt()));
        final int leaps = data.getInt();
        builder
            .leapSeconds(leaps)
            .leapIsValid(getBoolean(data))
            .startTime(fromDcTime64(dcTime64Start, leaps));
        final int postPtp = data.position();
        // Now go back to the Hall transitions.
        data.position(tranStart);

        // Gather the Hall transitions.
        LOG.log(Level.FINER, "[Raw Hall transitions] id, time, pos, on");
        for (int i = 0; i < ntrans; ++i) {
            builder.addHallTransition(decodeHallTransition(data, leaps));
        }
        LOG.log(Level.FINER, "[End Hall]");
        // Skip the unused elements of the transition array.
        data.position(tranStart + maxtrans * HALL_TRANSITION_SIZE);
       
        // Gather the encoder samples.
        for (int i = 0; i < nsamples; ++i) {
            builder.addEncoderSample(decodeEncoderSample(data, leaps));
        }        
        // Skip unused sample elements.
        data.position(data.position() + (maxsamples - nsamples) * ENCODER_SAMPLE_SIZE);
        
        
        // Finally ...
        data.position(postPtp);  // Don't want complaints about unused data.
        this.motion = builder.build();                
    }

    @Override
    public void encode(final ByteBuffer data) {
        super.encode(data);
        final Axis ax = motion.side() == PLUSX ? Axis.getPlusXSide() : Axis.getMinusXSide();
        data.putInt(ax.getPLCAxisNum());
        data.putLong(toDcTime64(motion.startTime()));
        final int tdai = (int)Duration
            .between(motion.startTime().getUTCInstant(), motion.startTime().getTAIInstant())
            .getSeconds();
        data.putDouble(motion.startPosition());
        data.putLong(toDcDuration(motion.targetDuration()));
        data.putDouble(motion.targetPosition());
        data.putDouble(motion.endPosition());
        data.putLong(toDcDuration(motion.actualDuration()));
        final int ntrans = motion.hallTransitions().size();
        final int unused = 5;
        data.putInt(ntrans);
        data.putInt(ntrans + unused);
        final int nsamples = motion.encoderSamples().size();
        data.putInt(nsamples);
        data.putInt(nsamples + unused);
        for (final HallTransition trans: motion.hallTransitions()) {
            encodeHallTransition(data, trans, tdai);
        }
        // Skip space before the encoder samples, simulating unused array entries.
        data.position(data.position() + unused * HALL_TRANSITION_SIZE);

        for (final EncoderSample sample: motion.encoderSamples()) {
            encodeEncoderSample(data, sample, tdai);
        }
        
        // Unused encoder sample space.
        data.position(data.position() + unused * ENCODER_SAMPLE_SIZE);
        
        // PTP-related data.
        data.putInt(PtpDeviceState.toStateNumber(PtpDeviceState.SLAVE));
        data.putInt(tdai); // leapSeconds.
        putBoolean(data, true);  // leapIsValid.
    }
    
    private HallTransition decodeHallTransition(final ByteBuffer data, final int leapSeconds) {
        final int id = data.getInt();
        final long rawtime = data.getLong();
        final CCSTimeStamp time = fromDcTime64(rawtime, leapSeconds);
        final double pos = data.getDouble();
        final boolean on = getBoolean(data);
        LOG.log(Level.FINER, "{0}, {1}, {2}, {3}", new Object[]{id, Long.toUnsignedString(rawtime), pos, on});
        return new HallTransition(time, id, pos, on);
    }
    
    // Encodes and returns size in bytes.
    private static int encodeHallTransition(
        final ByteBuffer data, final HallTransition hall, final int leapSeconds)
    {
        final int start = data.position();
        data.putInt(hall.getSensorId());
        data.putLong(toDcTime64(hall.getTime()));
        data.putDouble(hall.getPosition());
        putBoolean(data, hall.isOn());
        return data.position() - start;
    }
    
    private static EncoderSample decodeEncoderSample(final ByteBuffer data, final int leapSeconds) {
        final CCSTimeStamp time = fromDcTime64(data.getLong(), leapSeconds);
        final double pos = data.getDouble();
        return new EncoderSample(time, pos);
    }
    
    private static int encodeEncoderSample(
        final ByteBuffer data, final EncoderSample sample, final int leapSeconds)
    {
        final int start = data.position();
        data.putLong(toDcTime64(sample.getTime()));
        data.putDouble(sample.getPosition());
        return data.position() - start;
    }
    
    // Figure out how many bytes a Hall transition takes in a PLC message by encoding a dummy instance.
    // The same for an encoder sample.
    private static final int HALL_TRANSITION_SIZE;
    private static final int ENCODER_SAMPLE_SIZE;
    static {
        final ByteBuffer data = ByteBuffer.allocate(1000);
        final HallTransition hall = new HallTransition(CCSTimeStamp.currentTime(), 0, 0.0, false);
        HALL_TRANSITION_SIZE = encodeHallTransition(data, hall, 0);
        final EncoderSample sample = new EncoderSample(CCSTimeStamp.currentTime(), 0.0);
        ENCODER_SAMPLE_SIZE = encodeEncoderSample(data, sample, 0);
    }
    

    /**
     * Creates a string of the form 'MotionDonePLC{MsgToCCS={...} motion=MotionDone{...}}'
     * @return 
     */
    @Override
    public String toString() {
        return "MotionDonePLC{" + super.toString() + " motion=" + motion.toString() + '}';
    }
    
    /**
     * Returns the internal {@code MotionDone} instance.
     * @return The {@code MotionDone} instance.
     */
    public MotionDone getStatusBusMessage() {return motion;}

}
