package org.lsst.ccs.subsystem.shutter.plc;


import org.lsst.ccs.subsystem.shutter.common.Axis;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.Map;
import java.util.function.DoubleSupplier;
import java.util.stream.Collectors;
import java.util.stream.DoubleStream;

/**
 * Contains a CalibDone message sent from the PLC.
 * @author tether
 */
public class CalibDone extends MsgToCCS {

    /**
     * The number of transitions per direction of travel.
     */
    private static int TRANSITION_COUNT = 240;

    private final Map<Axis, Double> homePosition;
    private final Map<Axis, Double> deployedPosition;
    private final Map<Axis, Map<Direction, double[]>> axisTransitions;

    /**
     * Creates the message from scratch.
     * @param sequence A message sequence number.
     * @param homePosition Axis blade set home positions (home = fully retracted).
     * @param deployedPosition Axis blade set deployed positions (fully extended).
     * @param supplier The supplier of double values for the Hall calibration. Used to fill the
     * internal map of Hall transition positions which has "dimensions"
     * [axis number][direction][transitions per direction].
     * The order is row-major, Axis 0 comes before Axis 1, positive direction comes before negative. 240
     * values are expected for each combination of axis and direction.
     * @see Axis
     * @see Direction
     */
    public CalibDone(final int sequence,
                     final double[] homePosition,
                     final double[] deployedPosition,
                     final DoubleSupplier supplier)
    {
        super(sequence);
        this.homePosition = new EnumMap<>(Axis.class);
        this.homePosition.put(Axis.AXIS0, homePosition[0]);
        this.homePosition.put(Axis.AXIS1, homePosition[1]);
        this.deployedPosition = new EnumMap<>(Axis.class);
        this.deployedPosition.put(Axis.AXIS0, deployedPosition[0]);
        this.deployedPosition.put(Axis.AXIS1, deployedPosition[1]);
        this.axisTransitions = makeTransitionMap(supplier);
    }

    /**
     * Reads and converts the PLC form of the message.
     * @param data The PLC message data.
     */
    public CalibDone(final ByteBuffer data) {
        super(data);
        this.homePosition = new EnumMap<>(Axis.class);
        this.homePosition.put(Axis.AXIS0, data.getDouble());
        this.homePosition.put(Axis.AXIS1, data.getDouble());
        this.deployedPosition = new EnumMap<>(Axis.class);
        this.deployedPosition.put(Axis.AXIS0, data.getDouble());
        this.deployedPosition.put(Axis.AXIS1, data.getDouble());
        final DoubleSupplier supplier = () -> data.getDouble();
        this.axisTransitions = makeTransitionMap(supplier);
    }

    private static Map<Axis, Map<Direction, double[]>> makeTransitionMap(final DoubleSupplier supplier) {
        final Map<Axis, Map<Direction, double[]>> axisTransitions = new EnumMap<>(Axis.class);
        for (final Axis ax: Axis.values()) {
            final Map<Direction, double[]> directions = new EnumMap<>(Direction.class);
            for (final Direction di: Direction.values()) {
                final double[] trans = new double[TRANSITION_COUNT];
                for (int i = 0; i < trans.length; ++i) {
                    trans[i] = supplier.getAsDouble();
                }
                directions.put(di, trans);
            }
            axisTransitions.put(ax, directions);
       }
        return axisTransitions;
    }

    @Override
    public void encode(final ByteBuffer data) {
        super.encode(data);
        for (final Axis ax : Axis.values()) {
            data.putDouble(homePosition.get(ax));
        }
        for (final Axis ax : Axis.values()) {
            data.putDouble(deployedPosition.get(ax));
        }
        for (final Axis ax : Axis.values()) {
            for (final Direction di : Direction.values()) {
                final double[] trans = getHallTransitions(ax, di);
                for (int i = 0; i < trans.length; ++i) {
                    data.putDouble(trans[i]);
                }
            }
        }
    }

    /**
     * Returns the set of Hall transition positions for a given axis and direction of travel.
     * @param axis The axis.
     * @param direction The direction of travel.
     * @return The array of coordinates of the requested Hall transition.
     */
    public double[] getHallTransitions(final Axis axis, final Direction direction) {
        return axisTransitions.get(axis).get(direction);
    }

    /**
     * Returns the home position for an axis.
     * @param ax The axis.
     * @return The home position calculated from the axis homing routine.
     */
    public double getHomePosition(final Axis ax) {
        return homePosition.get(ax);
    }

    /**
     * Returns the deployed position for an axis.
     * @param ax The axis.
     * @return The deployed position calculated from the axis homing routine.
     */
    public double getDeployedPosition(final Axis ax) {
        return deployedPosition.get(ax);
    }

    @Override
    public String toString() {
        final StringBuffer buf = new StringBuffer("CalibDone{" + super.toString());
        for (final Axis ax: Axis.values()) {
            buf.append(", home[");
            buf.append(ax);
            buf.append("]=");
            buf.append(homePosition.get(ax));
        }
        for (final Axis ax: Axis.values()) {
            buf.append(", deployed[");
            buf.append(ax);
            buf.append("]=");
            buf.append(deployedPosition.get(ax));
        }
        for (final Axis ax: Axis.values()) {
            for (final Direction di: Direction.values()) {
                final DoubleStream pos = Arrays.stream(getHallTransitions(ax, di));
                buf.append(", transitions[");
                buf.append(ax);
                buf.append("][");
                buf.append(di);
                buf.append("]={");
                buf.append(pos.mapToObj(Double::toString).collect(Collectors.joining(", ")));
                buf.append("}");
            }
        }
        buf.append("}");
        return buf.toString();
    }

}
