package org.lsst.ccs.subsystem.shutter.mockquad;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.power.constants.PowerAgentProperties;

public class MockQuad extends Subsystem implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.TOP)
    protected volatile Subsystem subsys;
        
    private static final String BRAKE_VOLTAGE_KEY = "PDU_24VD/Shtr_Brakes_V";
    
    private static final Alert POWER_ALERT = new Alert(BRAKE_VOLTAGE_KEY, "Shutter brake power status.");
    
    public MockQuad() {
        super("mock-quadbox", AgentInfo.AgentType.WORKER);
    }
    
    /**
     ***************************************************************************
     **
     ** Initializes the subsystem. *
     * **************************************************************************
     */
    
    @Override
    public void init() {
        subsys
            .getAgentService(AlertService.class)
            .registerAlert(POWER_ALERT, ClearAlertHandler.ALWAYS);
        final KeyValueData msg = new KeyValueData(BRAKE_VOLTAGE_KEY, 0.0);
        subsys
            .getAgentService(DataProviderDictionaryService.class)
            .registerData(msg);
    }

    @Override
    public void postInit() {
        subsys
            .getAgentService(AgentPropertiesService.class)
            .setAgentProperty(PowerAgentProperties.QUAD_BOX_AGENT, this.getClass().getSimpleName());
    }
    
    /**
     * Stops listening for power status messages.
     */
    @Override
    public void postShutdown() {
    }
    
    @Command(type = CommandType.QUERY, description = "Raise the power alert at severity WARNING.")
    public void raiseAlert() {
        subsys.getAgentService(AlertService.class).raiseAlert(POWER_ALERT, AlertState.WARNING, "Brake power lost.");
    }
    
    @Command(type = CommandType.QUERY, description = "Publish a new shutter brake power voltage.")
    public void publishVoltage(final double v) {
        final KeyValueData kvd = new KeyValueData(BRAKE_VOLTAGE_KEY, v);
        final KeyValueDataList kvdl = new KeyValueDataList();
        kvdl.addData(kvd);
        subsys.publishSubsystemDataOnStatusBus(kvdl);
    }
}
