package org.lsst.ccs.subsystem.shutter;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import static org.lsst.ccs.bus.states.AlertState.NOMINAL;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TOP;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.power.constants.PowerAgentProperties;
import org.lsst.ccs.subsystem.shutter.statemachine.Channel;
import org.lsst.ccs.subsystem.shutter.statemachine.EventReply;
import org.lsst.ccs.subsystem.shutter.statemachine.SynchronousChannel;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * Generates {@code brakePowerChange()} events for the central state machine based on status messages
 * sent by a target subsystem, normally the quadbox subsystem (see org-lsst-ccs-subsystem-power).
 * We assume that brake power has been lost if the target subsystem does any of the following:<br>
 * <ulist>
 *     <li>Goes off-line.</li>
 *     <li>Sends a power status message in which the shutter brake voltage is too low.</li>
 *     <li>Raises a shutter brake power-loss alert of level higher than NOMINAL.</li>
 * </ulist>
 * <p>
 * We assume that brake power has been restored if the target subsystem:<br>
 * <ulist>
 *     <li>Sends a power status message in which the shutter brake voltage is high enough.</li>
 *     <li>Clears an outstanding shutter brake power-loss alert or reduces its level to NOMINAL.</li>
 * </ulist>
 * <p>It's not enough for the target subsystem to be on-line; we need to get a positive indication that
 * brake power is on.
 * <p>
 * The {@code brakeVoltageKey} configuration parameter contains the name of the PDU channel supplying
 * the voltage. It's used as the key for the voltage value in the {@code KeyValueDataList} published by the
 * quadbox subsystem periodically. It's also use as the ID of the alert raised to signal problems with
 * the voltage.
 * <p> The quadbox subsystem is recognized not by name but by it's having an agent property named
 * {@code PowerAgentProperties.QUAD_BOX_AGENT} which is transmitted with all of its messages.
 * <p>
 * Since the callbacks used to monitor the status bus should not block and the order of events must be
 * preserved, we add power-is-on flags to an unbounded queue which is read by a dedicated task. The task
 * undertakes to send the required events to the central state machine and waits for the state machine
 * to reply to each event before sending the next.
 * @author tether
 */
public class BrakePowerMonitor implements HasLifecycle, StatusMessageListener, AlertListener {

    private static final Logger LOG = Logger.getLogger(BrakePowerMonitor.class.getName());

    @LookupField(strategy = TOP)
    private volatile Subsystem subsys;
    
    @LookupField(strategy = TREE)
    private volatile StateMachine machine;
    
    private volatile AgentMessagingLayer aml;
    
    @ConfigurationParameter(description = "The key of the brake voltage value in the KeyValueDataList.",
        isFinal = true, units="unitless")
    private volatile String brakeVoltageKey;
    
    @ConfigurationParameter(
        description = "Brake power voltage must be at least this to be considered 'on'.",
        isFinal = true, units = "V")
    private volatile double powerOnThreshold;
    
    private final BlockingQueue<Boolean> eventQueue;
    
    private final Scheduler eventScheduler;
    
    BrakePowerMonitor() {
        brakeVoltageKey = "PDU_24VD/Shtr_Brakes_V";
        powerOnThreshold = 23.0;
        eventQueue = new LinkedBlockingQueue<>();
        eventScheduler = new Scheduler("brakePower", 1, new ThreadGroup("Brake power monitoring"));
    }
    
    ////////// Implementation of HasLifeCycle //////////
    /**
     * Starts listening for alerts from the target subsystem.
     */
    @Override
    public void init() {
        subsys.getAgentService(AlertService.class).addListener(this);
        subsys.getAgentService(AlertService.class).startStatusAlertListening((AgentInfo a) -> true);
    }
    
    /**
     * Starts listening for power status messages from the publisher.
     */
    @Override
    public void postStart() {
        aml = subsys.getMessagingAccess();
        aml.addStatusMessageListener(
            this,     // Call our onStatusMessage() method.
            msg ->
                msg.getOriginAgentInfo().hasAgentProperty(PowerAgentProperties.QUAD_BOX_AGENT)
                && msg instanceof StatusSubsystemData
                && ((StatusSubsystemData)msg).getObject() instanceof KeyValueDataList
        );
        // Must not start this until the state machine is running. postStart() is the last of the
        // startup lifecycle methods to be called.
        eventScheduler.schedule(this::schedulerLoop, 0/*delay*/, TimeUnit.SECONDS);
    }
    
    /**
     * Stops listening for power status messages.
     */
    @Override
    public void postShutdown() {
        eventScheduler.shutdownNow();
        aml.removeStatusMessageListener(this);
        subsys.getAgentService(AlertService.class).removeListener(this);
    }
    
    ////////// Implementation of StatusMessageListener //////////

    @Override
    public void onStatusMessage(final StatusMessage rawMsg) {
        final List<KeyValueData> kvdList =
            ((KeyValueDataList)((StatusSubsystemData)rawMsg).getObject()).getListOfKeyValueData();
        Optional<KeyValueData> kvd =
            kvdList.stream()
                .filter(k -> k.getKey().equals(brakeVoltageKey))
                .findFirst();
        if (kvd.isPresent()) {
            final double voltage = (Double)kvd.get().getValue();
            sendPowerEvent( voltage >= powerOnThreshold );
        }
    }
    
    ////////// Implementation of AlertListener //////////
        
    @Override
    public void onAlert(final AlertEvent event) {
        switch (event.getType()) {
            case AGENT_CONNECTION:
                break;
            case AGENT_DISCONNECTION:
                sendPowerEvent(false);
                break;
            case ALERT_CLEARED:
                {
                    // Many alerts may get cleared all at once.
                    final List<String> clearedIds = event.getClearedIds();
                    sendPowerEvent( clearedIds.contains(brakeVoltageKey) );
                }
                break;
            case ALERT_RAISED:
                {
                    final Alert alrt = event.getAlert();
                    if (alrt.getAlertId().equals(brakeVoltageKey)) {
                        sendPowerEvent(event.getLevel() == NOMINAL);
                    }
                }
                break;
        }
    }

    
    ////////// Event-sending code //////////
    private void sendPowerEvent(final boolean powerIsOn) {
        eventQueue.add(powerIsOn);
    }
    
    private void schedulerLoop() {
        LOG.info("Starting the brake power monitoring task.");
        final Channel<EventReply> replyChan = new SynchronousChannel<>();
        try {
            for (;;) {
                final boolean powerIsOn = eventQueue.take();
                try {
                    machine.brakePowerChange(replyChan, powerIsOn);
                    // Must wait for reply in order to preserve order of events.
                    final EventReply reply = replyChan.read();
                }
                catch (InterruptedException exc) {
                    throw exc;
                }
                catch (Exception exc) {
                    LOG.log(Level.SEVERE, "Exception in brake power monitoring task.", exc);
                }
            }
        }
        catch (InterruptedException exc) {
            LOG.info("Normal termination of the brake power monitoring task.");
        }
    }

}
