package org.lsst.ccs.subsystem.shutter;

import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.services.alert.AlertService;

/**
 * All the alerts that this subsystem can raise.
 * @author tether
 */
public enum Alerts {

    /**
     * Raised if the PLC state machine and the subsystem state machine appear to
     * have gotten out of sync. 
     */
    SYNC(new Alert("SYNC", "Status of tracking of controller state.")),

    /**
     * Raised if the shutter controller sends an explicit error message to the subsystem.
     */
    PLC(new Alert("PLC", "Notice of errors from shutter controller.")),

    /**
     * Raised if no messages have arrived from the shutter controller within a certain period.
     */
    WATCHDOG(new Alert("WATCHDOG", "PLC message rate monitor.")),

    /**
     * Raised if the Hall data from the last motion indicate a significant deviation
     * from the predicted motion.
     */
    MOTION(new Alert("MOTION", "Check of motion profile using Hall data.")),
    
    /**
     * Raised if some problem has been observed with the motors. For example 
     * temperature out of the safe operating range.
     * 
     */
    MOTOR(new Alert("MOTOR", "A problem with the motors.")),
    
    /**
     * Raised if the subsystem has been given the stopAllMotion() command while in Prod.
     */
    STOP_CMD(new Alert("STOP_CMD",
        "Received the stopAllMotion() command while in the Prod superstate."));
    
    private final Alert ccsAlert;
    
    Alerts(final Alert ccsAlert) {this.ccsAlert = ccsAlert;}
    
    /**
     * Raises the alert associated with this enumeration member.
     * @param alertSvc the CCS alert service.
     * @param severity the state which the raised alert is to have.
     * @param cause the cause string which the raised alert is to have.
     */
    public void raise(
        final AlertService alertSvc,
        final AlertState severity,
        final String cause,
        final AlertService.RaiseAlertStrategy strategy)
    {
        alertSvc.raiseAlert(ccsAlert, severity, cause, strategy);
    }
    
    public static void registerAll(final AlertService alertSvc) {
        for (final Alerts a: Alerts.values()) {
            alertSvc.registerAlert(a.ccsAlert, AlertService.ALWAYS);
        }
    }

}
