package org.lsst.ccs.subsystem.shutter;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.subsystem.common.service.DataAccumulationService;

/**
 * Maintains persistent counters that are also trended. Uses both the
 * data accumulation and the trending services.
 * @author tether
 */
public class PersistentCounter {
    
    private final String path;
    private final Subsystem subsys;
    private final DataAccumulationService accumSvc;

    private PersistentCounter(final String path, final Subsystem subsys) {
        this.path = path;
        this.subsys = subsys;
        this.accumSvc = subsys.getAgentService(DataAccumulationService.class);
    }
    
    /**
     * Register the path with {@code DataProviderDictionaryService} and with {@code DataAccumulationService}.
     */
    private void register() {
        final DataProviderDictionaryService dataSvc
            = subsys.getAgentService(DataProviderDictionaryService.class);
        dataSvc.registerData(new KeyValueData(path, 0.0));
        accumSvc.registerAccumulatedDataPath(path);
    }
    
    /**
     * Makes a new persistent counter. Must be called during the init() stage of the subsystem's life cycle.
     * @param path The path name.
     */
    public static PersistentCounter newCounter(final String path, final Subsystem subsys) {
        final PersistentCounter ctr = new PersistentCounter(path, subsys);
        ctr.register();
        return ctr;
    }
    
    /**
     * Adds to the counter's current value and publishes the new sum (if available).
     * @param inc What to add to the counter value.
     */
    public void increment(final double inc) {
        subsys.getAgentService(DataAccumulationService.class).accumulateData(path, inc);
        final double sum = getValue();
        if (!Double.isNaN(sum)) {
            subsys.publishSubsystemDataOnStatusBus(new KeyValueData(path, sum));
        }
    }
    
    /**
     * Gets the current value of the counter.
     * @return The value, or NaN if the REST file server is unreachable.
     */
    public double getValue() {
        return accumSvc.getAccumulatedValueForPath(path);
    }

}
