package org.lsst.ccs.subsystem.shutter.statemachine;

import java.util.logging.Logger;
import org.lsst.ccs.subsystem.shutter.plc.Ignored;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEntry;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEvent;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logExit;
import static org.lsst.ccs.subsystem.shutter.statemachine.PromptReply.ACCEPTED;

/**
 * State in which the state machines for the CCS subsystem
 * and the PLC are known to be in compatible states. Thread-safe.
 * @author tether
 */
class InSync extends CompositeState<InSync, TopContext> {
    private static final Logger LOG = Logger.getLogger(InSync.class.getName());

    // PROTECTED by the instance lock.
    private Enabled enabledState;
    private Disabled disabledState;
    // END PROTECTED

    /**
     * Stores the given action implementations and context for this composite state.
     * @param actions The set of action implementations to pass to substates.
     * @param context The super-context for this context.
     * @throws NullPointerException if either argument is null.
     */
    InSync(final Actions actions, final TopContext context)
    {
        super(actions, context);
    }

    @Override
    public Logger getLogger() {return LOG;}

    synchronized Enabled getEnabledState() {
        if (enabledState == null) enabledState = new Enabled(getActions(), this);
        return enabledState;
    }

    synchronized Disabled getDisabledState() {
        if (disabledState == null) disabledState = new Disabled(this);
        return disabledState;
    }

    /**
     * @implNote Sets the state to either {@code Enabled} or {@code Disabled}
     * depending on the result of calling {@code getPLCEnabled()} in this state's context.
     */
    @Override
    public void init() {
        final State<InSync> initState =
                getContext().getPLCEnabled() ? getEnabledState() : getDisabledState();
        makeTransition(initState, null);
    }

    /**
     * {@inheritDoc } Changes the SYNC alert level to NOMINAL so that it may be cleared.
     */
    @Override
    public void entry() {
        logEntry(this);
        getContext().getActions().lowerSyncAlert();
        // If makeFullContact() fails it will post a contactLost() event.
        getContext().getActions().makeFullContact();
        // Start counting messages from the PLC.
        getContext().getActions().enableWatchdog();
        init();
    }

    /**
     * {@inheritDoc} Disables the PLC message watchdog.
     */
    @Override
    public void exit() {
        getContext().getActions().disableWatchdog();
        logExit(this);
    }
    @Override
    public void contactLost(final Channel<EventReply> chan) throws InterruptedException {
        logEvent(this);
        chan.write(ACCEPTED);
        getContext().makeTransition(getContext().getSyncErrorState(), null);
    }

    @Override
    public void ignored(final Channel<EventReply> chan, Ignored.Reason reason) throws InterruptedException {
        logEvent(this);
        chan.write(ACCEPTED);
        LOG.warning(() -> String.format("The shutter controller sent the ignored(%s) event.", reason));
        getContext().makeTransition(getContext().getSyncErrorState(), null);
    }

}
