package org.lsst.ccs.subsystem.shutter;

import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.alert.AlertService;
import static org.lsst.ccs.services.alert.AlertService.RaiseAlertStrategy.ON_SEVERITY_CHANGE;
import org.lsst.ccs.subsystem.shutter.statemachine.SynchronousChannel;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * Keeps track of the number of PLC messages received and if none are received within a configurable interval,
 * raise a PLC_WATCHDOG alert of severity WARNING. A small state machine is run by its own thread. The initial state
 * is DISABLED, in which any event but enable() is ignored. In the ENABLED state, the countMessage() event causes
 * the event counter to be incremented and the checkMessageCount() event causes a check to be made of
 * the current message count. If the count is zero then a WARNING alert is raised and the count is reset to zero. The count is
 * also reset upon entry to the DISABLED state  .
 * <p>
 * @author tether
 */
public class Watchdog  implements HasLifecycle {
    private static final Logger LOG = Logger.getLogger(Watchdog.class.getName());

    @LookupField(strategy = LookupField.Strategy.TREE)
    private volatile Subsystem subsys;

    @ConfigurationParameter(description="The interval between checks of the PLC  message counter.", units="s")
    private volatile Duration checkInterval;

    private static enum WatchState {ENABLED, DISABLED;}

    private int msgCount;
    private final SynchronousChannel<Runnable> inChannel;
    private WatchState state;
    private final Scheduler sched;

    ////////// Lifecycle methods //////////

    public Watchdog() {
        this.msgCount = 0;
        this.inChannel = new SynchronousChannel<>();
        this.state = WatchState.DISABLED;
        this.sched = new Scheduler("PLCWatchdog", 2);
        this.checkInterval = Duration.ofSeconds(10);
    }

    @Override
    public void start() {
        sched.scheduleWithFixedDelay(this::watchdogBody, 0, 10, TimeUnit.MILLISECONDS);
        sched.scheduleAtFixedRate(this::timerBody, 100, checkInterval.toMillis(), TimeUnit.MILLISECONDS);
    }

    @Override
    public void shutdown() {
        sched.shutdownNow();
    }

    ////////// Public interface when in operation //////////

    /**
     * If this is in the disabled state, switch to the enabled state.
     * @throws InterruptedException
     */
        public void enable() throws InterruptedException {
        inChannel.write(this::enableImpl);
    }

    /**
     * If this is in the enabled state, switch to the disabled state and set the count of messages received to zero.
     * @throws InterruptedException
     */
    public void disable() throws InterruptedException {
        inChannel.write(this::disableImpl);
    }

    /**
     * If this is in the enabled state, increment the count of messages received.
     * @throws InterruptedException
     */
    public void countMessage()  throws InterruptedException {
        inChannel.write(this::countImpl);
    }

    /**
     * If this is in the enabled state, raise an alert if no messages have yet been received and then reset
     * the count of messages received.
     * @throws InterruptedException
     */
    public void checkMessageCount() throws InterruptedException {
        inChannel.write(this::checkImpl);
    }


    ////////// Private implementation //////////
    private void enableImpl() {
        if (state == WatchState.DISABLED) {state = WatchState.ENABLED;}
    }

    private void disableImpl() {
        if (state == WatchState.ENABLED) {
            state = WatchState.DISABLED;
            msgCount = 0;
        }
    }

    private void countImpl() {
        if (state == WatchState.ENABLED) {++msgCount;}
    }

    private void checkImpl() {
        if (state == WatchState.ENABLED) {
            if (msgCount == 0) {
                subsys.getAgentService(AlertService.class)
                    .raiseAlert(Alerts.WATCHDOG,
                        AlertState.WARNING,
                        "No messages are coming in from the shutter.",
                        ON_SEVERITY_CHANGE);
            }
            else {
                msgCount = 0;
            }
        }
    }

    private void watchdogBody() {
        try {
            Thread.currentThread().setName("Watchdog body");
            LOG.info("Starting the PLC message watchdog.");
            while (true) {inChannel.read().run();}
        }
        catch (InterruptedException exc) {
            LOG.info("Normal exit of the PLC message watchdog.");
        }
    }

    private void timerBody() {
        try {
            checkMessageCount();
        }
        catch (InterruptedException exc) {
            LOG.info("The PLC message watchdog timer was interrupted!");
        }
    }
}
