package org.lsst.ccs.subsystem.shutter.statemachine;

import java.time.Duration;
import java.util.Objects;

/**
 * Response, available without blocking, to an event. Thread-safe.
 * @author tether
 */
final class PromptReply implements EventReply {

    /** The instance always used when an event is accepted. */
    public static final PromptReply ACCEPTED = new PromptReply();

    private final String msg;
    private final boolean accepted;

    private volatile boolean waited;

    /**
     * Creates an acceptance message.
     */
    private PromptReply() {
        this.accepted = true;
        this.msg = "ACCEPTED";
        this.waited = false;
    }

    /**
     * Creates a rejection message.
     * @param msg The message describing the rejection.
     * @throws NullPointerException if {@code msg} is null.
     */
    public PromptReply(final String msg) {
        this.accepted = false;
        this.msg = Objects.requireNonNull(msg, "Argument must not be null.");
        this.waited = false;
    }


    /**
     * Gets the acceptance flag. Never blocks or throws.
     * @return {@inheritDoc }
     */
    @Override
    public boolean wasAccepted(final Duration timeout) {
        waited = true;
        return accepted;
    }

    /**
     * @throws IllegalStateException if {@code wasAccepted() wasn't called.}
     * @return {@inheritDoc}
     * @see #wasAccepted() 
     */
    @Override
    public String getMessage() {
        if (!waited) {throw new IllegalStateException("Didn't wait for completion with wasAccepted().");}
        return msg;
    }

    /**
     * Gets a readable string describing the reply.
     * @return A string of the form {@code PromptReply{message-text}}
     */
    @Override
    public String toString() {
        return "PromptReply{" + msg + '}';
    }

}
