package org.lsst.ccs.subsystem.shutter;

import java.time.Duration;
import java.util.logging.Level;
import java.util.stream.Collectors;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.shutter.common.EncoderSample;
import org.lsst.ccs.subsystem.shutter.common.HallTransition;
import org.lsst.ccs.subsystem.shutter.common.PLCError;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.Error;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import org.lsst.ccs.subsystem.shutter.plc.MsgToPLC;
import org.lsst.ccs.subsystem.shutter.statemachine.Actions;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * An implementation of the {@code Actions} interface that forwards most actions to
 * the {@code StateMachine} component or to the {@code Controller} component. A few actions
 * are implemented directly, mainly those which make calls to the subsystem framework that
 * don't affect either the state machine or the shutter PLC. Immutable.
 * @see StateMachine
 * @see Controller
 * @see Actions
 * @author tether
 */
public class RealActions implements Actions {
    private static final Logger LOG = Logger.getLogger(RealActions.class.getName());

    private final Controller control;
    private final StateMachine machine;
    private final Subsystem subsys;
    private final Publisher publish;
    private final Watchdog wdog;

    /**
     * Saves references to the components that actually implement the actions.
     * @param control A references to the {@code Controller} component.
     * @param machine A reference to the {@code StateMachine} component.
     * @param subsys A reference to the subsystem object.
     * @param publish A reference to the {@code Publisher} component.
     * @param wdog A reference to the {@code Watchdog} component.
     */
    public RealActions(
        final Controller control,
        final StateMachine machine,
        final Subsystem subsys,
        final Publisher publish,
        final Watchdog wdog)
    {
        this.control = control;
        this.machine = machine;
        this.subsys = subsys;
        this.publish = publish;
        this.wdog = wdog;
    }

    @Override
    public boolean axisIsEnabled(final ShutterSide side) {
        final ShutterStatus status = publish.getShutterStatus();
        return status.getAxisStatus(side).isEnabled();
    }

    @Override
    public boolean brakeIsEngaged(final ShutterSide side) {
        final ShutterStatus status = publish.getShutterStatus();
        return status.getAxisStatus(side).isBrakeEngaged();
    }

    @Override
    public void cancelSyncTimer() {
        LOG.fine("{ACTION cancelSyncTimer()}");
        machine.cancelSyncTimer();
    }

    @Override
    public void disableWatchdog() {
        LOG.fine("{ACTION disableWatchdog()}");
        try {
             wdog.disable();
        }
        catch (InterruptedException exc) {
            LOG.info("Watchdog disable was interrupted.");
            Thread.currentThread().interrupt();
        }
    }

    @Override
    public void enableWatchdog() {
        LOG.fine("{ACTION enableWatchdog()}");
        try {
             wdog.enable();
        }
        catch (InterruptedException exc) {
            LOG.info("Watchdog enable was interrupted.");
            Thread.currentThread().interrupt();
        }
    }

    @Override
    public boolean isBadExposureTime(final Duration exposureTime) {
        LOG.fine("{ACTION isBadExposure()}");
        return control.isBadExposureTime(exposureTime);
    }

    @Override
    public void lowerMotionAlert() {
        LOG.fine("{ACTION lowerMotionAlert()}");
        subsys.getAgentService(AlertService.class).raiseAlert(Alerts.MOTION, AlertState.NOMINAL,
            "PLC reset/resync.");
    }

    @Override
    public void lowerPLCAlert() {
        LOG.fine("{ACTION lowerPLCAlert()}");
        subsys.getAgentService(AlertService.class).raiseAlert(Alerts.PLC, AlertState.NOMINAL,
            "PLC reset/resync.");
    }

    @Override
    public void lowerSyncAlert() {
        LOG.fine("{ACTION lowerSyncAlert()}");
        subsys.getAgentService(AlertService.class)
                .raiseAlert(Alerts.SYNC, AlertState.NOMINAL, "Synchronized with PLC.");
    }

    @Override
    public void lowerWatchdogAlert() {
        LOG.fine("{ACTION lowerWatchdogAlert()}");
        subsys.getAgentService(AlertService.class)
                .raiseAlert(Alerts.WATCHDOG, AlertState.NOMINAL, "PLC reset/resync.");
    }

    @Override
    public boolean makePartialContact() {
        LOG.fine("{ACTION makePartialContact()}");
        return control.makePartialContact();
    }

    @Override
    public boolean makeFullContact() {
        LOG.fine("{ACTION makeFullContact()}");
        return control.makeFullContact();
    }

    @Override
    public void publishEnableStatus(final boolean isEnabled) {
        publish.setEnableStatus(isEnabled);
    }

    @Override
    public void setPhysicalState(final PhysicalState newState) {
        LOG.fine(String.format("{ACTION setPhysicalState(%s)}", newState));
        machine.setPhysicalState(newState);
    }

    @Override
    public void setSoftwareState(SoftwareState newState) {
        LOG.fine(String.format("{ACTION setSoftwareState(%s)}", newState));
        machine.setSoftwareState(newState);
    }

    @Override
    public void raisePLCAlert(final Error err) {
        LOG.fine("{ACTION raisePLCAlert()}");
        for (final PLCError e: err.getErrors()) {
            LOG.log(Level.WARNING, "PLC error {0} - {1}.", new Object[]{e.toString(), e.getDescription()});
        }
        final String errlist =
            err.getErrors().stream().map(PLCError::toString).collect(Collectors.joining(", "));
        subsys.getAgentService(AlertService.class).raiseAlert(Alerts.PLC, AlertState.ALARM, errlist);
    }

    @Override
    public void raiseSyncAlert() {
        LOG.fine("{ACTION raiseSyncAlert()}");
        subsys.getAgentService(AlertService.class).raiseAlert(Alerts.SYNC, AlertState.ALARM, "Lost track.");
    }

    @Override
    public boolean readyForCalibration() {
        LOG.fine("{ACTION readyForCalibration()}");
        return control.readyForCalibration();
    }

    @Override
    public void resetPLC() {
        LOG.fine("{ACTION resetPLC()}");
        control.resetPLC();
    }

    @Override
    public void relay(final MsgToPLC eventMsg) {
        LOG.log(Level.FINE, "{ACTION relay({0}}", eventMsg.getClass());
        control.relay(eventMsg);
    }

    @Override
    public void saveCalib(final CalibDone cal) {
        LOG.warn("{ACTION saveCalib() - not yet implemented.}");
        // TODO - saving a calibration.
    }

    @Override
    public void sendProfile(final MotionDonePLC motplc) {
        LOG.fine("{ACTION sendProfile()}");
        final MotionDone mot = motplc.getStatusBusMessage();
        if (LOG.isDebugEnabled()) {
            dumpMotionDone(mot);
        }
        control.checkHallTransitions(motplc);
        publish.publishMotionDone(mot);
    }

    private void dumpMotionDone(final MotionDone mot) {
        final StringBuilder bld = new StringBuilder();
        bld.append("[MotionDone begin]\n");
        // Header.
        bld.append("side " + mot.side() + "\n");
        bld.append("startPosition " + mot.startPosition() + " mm\n");
        bld.append("startTime " + mot.startTime().getUTCInstant().toEpochMilli() + " Unix epoch ms\n");
        bld.append("targetPosition " + mot.targetPosition() + " mm\n");
        bld.append("targetDuration " + mot.targetDuration().toMillis() + " ms\n");
        bld.append("endPosition " + mot.endPosition() + " mm\n");
        bld.append("actualDuration " + mot.actualDuration().toMillis() + " ms\n");
        // Hall transitions.
        bld.append("[Hall begin] time(Unix epoch ms) ID position(mm) isOn\n");
        for (final HallTransition hall: mot.hallTransitions()) {
            bld.append(hall.getTime().getUTCInstant().toEpochMilli());
            bld.append(" ");
            bld.append(hall.getSensorId());
            bld.append(" ");
            bld.append(hall.getPosition());
            bld.append(" ");
            bld.append(hall.isOn());
            bld.append("\n");
        }
        bld.append("[Hall end]\n");
        // Encoder samples.
        bld.append("[Encoder begin] time(Unix epoch ms) position(mm)\n");
        for (final EncoderSample enc: mot.encoderSamples()) {
            bld.append(enc.getTime().getUTCInstant().toEpochMilli());
            bld.append(" ");
            bld.append(enc.getPosition());
            bld.append("\n");
        }
        bld.append("[Encoder end]\n");
        bld.append("[MotionDone end]\n");
        LOG.fine(bld.toString());
    }

    @Override
    public void startFirstCentering() {
        LOG.fine("{ACTION startFirstCentering()}");
        control.startFirstCentering();
    }

    @Override
    public void startSecondCentering() {
        LOG.fine("{ACTION startSecondCentering(};}");
        control.startSecondCentering();
    }

    @Override
    public String shutterIsReady() {
        LOG.fine("{ACTION shutterIsReady()}");
        return control.shutterIsReady();
    }

    @Override
    public void startSyncTimer() {
        LOG.fine("{ACTION startSyncTimer()}");
        machine.startSyncTimer();
    }

    @Override
    public void terminateContact() {
        LOG.fine("{ACTION terminateContact()}");
        control.terminateContact();
    }

}
