package org.lsst.ccs.subsystem.shutter;

import java.time.Duration;
import java.util.concurrent.Callable;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.RunMode;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.HomeAxis;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.Calibrate;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.Error;
import org.lsst.ccs.subsystem.shutter.plc.Ignored;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import org.lsst.ccs.subsystem.shutter.statemachine.Actions;
import org.lsst.ccs.subsystem.shutter.statemachine.BlackHoleChannel;
import org.lsst.ccs.subsystem.shutter.statemachine.Channel;
import org.lsst.ccs.subsystem.shutter.statemachine.EventReply;
import org.lsst.ccs.subsystem.shutter.statemachine.Events;
import org.lsst.ccs.subsystem.shutter.statemachine.SynchronousChannel;
import org.lsst.ccs.subsystem.shutter.statemachine.TopContext;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * Component that owns and operates the internal shutter-control state internalMachine. It also owns, registers
 * and updates a component state bundle based on classes {@code PhysicalState} and {@code SoftwareState}.
 * <p>
 * Used by commands in the {@code Main} and other components which call the event methods implemented here.
 * Those methods create callables for later execution. Each callable is passed an instance of
 * {@code Channel<EventReply>} which it uses to send event accept/reject notices back to the
 * command initiator.
 * @see Main
 * @see PhysicalState
 * @see SoftwareState
 * @see TopContext
 * @author tether
 */
public class StateMachine implements HasLifecycle, Events {
    private static final Logger LOG = Logger.getLogger(StateMachine.class.getName());

    /** A reference to the Subsystem component in the subsystem tree. */
    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Subsystem subsys;

    /** A reference to the {@code Controller} component that communicates with the shutter controller. */
    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Controller controller;

    /** A reference to the {@code Publisher} component that sends status bus messages. */
    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Publisher publish;

    /** A reference to the {@code Watchdog} component. */
    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Watchdog wdog;

    @ConfigurationParameter(description="How long to delay restarting a crashed event task.")
    private volatile Duration taskRestartDelay = Duration.ofMillis(10);

    @ConfigurationParameter(description="Wait this long for the shutter to enter Still or Disabled after a reset.")
    private volatile Duration resetSyncTimeout = Duration.ofSeconds(10);

    // GUARDED by the constructor.
    /** Use to pass events to the internal state internalMachine. */
    private final Channel<Callable<Void>> eventChan;

    /** The scheduler for the task that reads the event channel and for the sync timeout task. */
    private final Scheduler eventTaskScheduler;
    // END GUARDED

    /** Reads deferred events and feeds them to the internal state machine. */
    private volatile PeriodicTask eventTask;

    /** Implements the sync timeout for the {@code Synchronizing} state. */
    private volatile ScheduledFuture<?> syncTimeoutTask;

    /** The actual state internalMachine implementation. */
    private volatile TopContext internalMachine;

    /** The actions implementation used by the state internalMachine. */
    private volatile Actions actions;

    /**
     * Creates the state internalMachine implementation and the priority queue. The state internalMachine is not started.
     */
    public StateMachine() {
        this.eventChan = new SynchronousChannel<>();
        this.eventTaskScheduler = new Scheduler("State machine events", 3); // event task, timer, sim shutter.
        this.eventTaskScheduler.setLogger(LOG);
        this.eventTaskScheduler.setDefaultLogLevel(Level.SEVERE);
    }

    /**
     * Registers the shutter physical-state bundle and sets its initial value to {@code OTHER}.
     * @see PhysicalState
     */
    @Override
    public void init() {
        final AgentStateService stateServ = subsys.getAgentService(AgentStateService.class);
        stateServ.registerState(PhysicalState.class, "The physical state of the shutter", this);
        stateServ.updateAgentComponentState(this, PhysicalState.OTHER);
        stateServ.registerState(SoftwareState.class, "The state of the subsystem's central state machine", this);
        stateServ.updateAgentComponentState(this, SoftwareState.SYNCHRONIZING);
    }

    /**
     * Starts the shutter-control state internalMachine along with a task for feeding it events from the queue.
     */
    @Override
    public void postStart() {
        // The state internalMachine can't be created in the constructor because it would allow "this"
        // to escape and because the the controller reference would not have been set.
        if (RunMode.isSimulation()) {
            actions = new SimulatedActions(controller, this, subsys, publish, wdog);
            internalMachine = new TopContext(actions);
        }
        else {
            actions = new RealActions(controller, this, subsys, publish, wdog);
            internalMachine = new TopContext(actions);
        }
        internalMachine.init();
        eventTask =
            eventTaskScheduler.scheduleWithFixedDelay(this::eventTaskBody,
                                                      0, taskRestartDelay.toMillis(),
                                                      TimeUnit.MILLISECONDS);
        // If we're in simulated mode then put the state internalMachine into the Closed state.
        if (RunMode.isSimulation()) {
            try {
                final Channel<EventReply> replyChan = new SynchronousChannel<>();
                plcIsEnabled(replyChan);
                replyChan.read();
                gotoProd(replyChan);
                replyChan.read();
            } catch (InterruptedException ex) {
                LOG.log(Level.WARNING, null, ex);
            }
        }
    }

    /**
     * Gets the set of actions used by the internal state internalMachine.
     * @return The actions.
     */
    public Actions getActions() {return actions;}

    private void eventTaskBody() {
        LOG.info("The event task has started.");
        try {
            while (true) {
                LOG.fine("About to read an event.");
                final Callable<Void> event = eventChan.read();
                LOG.fine("Running an event.");
                event.call();
            }
        }
        catch (InterruptedException exc) {
            LOG.info("Normal stop of the event task.");
        }
        catch (Exception exc) {
            LOG.warning("Exception thrown in the internal state machine.", exc);
        }
    }

    @Override
    public void shutdown() {
        eventTaskScheduler.shutdownNow();
    }


    ////////// Events //////////

    @Override
    public void contactLost(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write(
                () -> {internalMachine.contactLost(replyChan); return null;}
        );
    }

    @Override
    public void plcIsEnabled(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write(
            () -> {internalMachine.plcIsEnabled(replyChan); return null;}
        );
    }

    @Override
    public void plcIsDisabled(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.plcIsDisabled(replyChan); return null;} );
    }

    @Override
    public void resync(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write(() -> {internalMachine.resync(replyChan); return null;});
    }

    @Override
    public void syncTimeout(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.syncTimeout(replyChan); return null;} );
    }

    @Override
    public void enable(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.enable(replyChan); return null;} );
    }

    @Override
    public void disable(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.disable(replyChan); return null;} );
    }

    @Override
    public void motionDone(final Channel<EventReply> replyChan, final MotionDonePLC profileData) throws InterruptedException {
        eventChan.write( () -> {internalMachine.motionDone(replyChan, profileData); return null;} );
    }

    @Override
    public void calibrate(final Channel<EventReply> replyChan, final Calibrate calibParams) throws InterruptedException {
        eventChan.write( () -> {internalMachine.calibrate(replyChan, calibParams); return null;} );
    }

    @Override
    public void calibDone(final Channel<EventReply> replyChan, final CalibDone calibResults) throws InterruptedException {
        eventChan.write( () -> {internalMachine.calibDone(replyChan, calibResults); return null;} );
    }

    @Override
    public void error(final Channel<EventReply> replyChan, final Error err) throws InterruptedException {
        eventChan.write( () -> {internalMachine.error(replyChan, err); return null;} );
    }

    @Override
    public void reset(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.reset(replyChan); return null;} );
    }

    @Override
    public void takeExposure(final Channel<EventReply> replyChan, final Duration exposureTime) throws InterruptedException {
        eventChan.write( () -> {internalMachine.takeExposure(replyChan, exposureTime); return null;} );
    }

    @Override
    public void openShutter(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.openShutter(replyChan); return null;} );
    }

    @Override
    public void timer(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.timer(replyChan); return null;} );
    }

    @Override
    public void closeShutter(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.closeShutter(replyChan); return null;} );
    }

    @Override
    public void ignored(final Channel<EventReply> replyChan, final Ignored.Reason reason) throws InterruptedException {
        eventChan.write( () -> {internalMachine.ignored(replyChan, reason); return null;} );
    }

    @Override
    public void gotoProd(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.gotoProd(replyChan); return null;} );
    }

    @Override
    public void gotoCenter(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.gotoCenter(replyChan); return null;} );
    }

    @Override
    public void moveAxisAbsolute(final Channel<EventReply> replyChan, final MoveAxisAbsolute req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.moveAxisAbsolute(replyChan, req); return null;} );
    }

    @Override
    public void moveAxisRelative(final Channel<EventReply> replyChan, final MoveAxisRelative req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.moveAxisRelative(replyChan, req); return null;} );
    }

    @Override
    public void clearAllFaults(final Channel<EventReply> replyChan, final ClearAllFaults req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.clearAllFaults(replyChan, req); return null;} );
    }

    @Override
    public void changeAxisEnable(final Channel<EventReply> replyChan, final ChangeAxisEnable req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.changeAxisEnable(replyChan, req); return null;} );
    }

    @Override
    public void changeBrakeState(final Channel<EventReply> replyChan, final Axis ax, final ChangeBrakeState.State newState) throws InterruptedException {
        eventChan.write( () -> {internalMachine.changeBrakeState(replyChan, ax, newState); return null;} );
    }

    @Override
    public void clearAxisFaults(final Channel<EventReply> replyChan, final ClearAxisFaults req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.clearAxisFaults(replyChan, req); return null;} );
    }

    @Override
    public void enableAllAxes(final Channel<EventReply> replyChan, final EnableAllAxes req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.enableAllAxes(replyChan, req); return null;} );
    }

    @Override
    public void disableAllAxes(final Channel<EventReply> replyChan, final DisableAllAxes req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.disableAllAxes(replyChan, req); return null;} );
    }

    @Override
    public void homeAxis(final Channel<EventReply> replyChan, final HomeAxis req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.homeAxis(replyChan, req); return null;} );
    }

    ////////// Actions //////////

    void setPhysicalState(final PhysicalState newState) {
        final AgentStateService stateServ = subsys.getAgentService(AgentStateService.class);
        stateServ.updateAgentComponentState(this, newState);
    }

    void setSoftwareState(final SoftwareState newState) {
        final AgentStateService stateServ = subsys.getAgentService(AgentStateService.class);
        stateServ.updateAgentComponentState(this, newState);
    }

    synchronized void startSyncTimer() {
        // We don't bother testing for rejection of the syncTimeout() event, which would happen
        // if we were late in canceling the timer task.
        final Channel<EventReply> chan = new BlackHoleChannel<>();
        final Callable<Void> snippet = () -> {this.syncTimeout(chan); return null;};
        syncTimeoutTask = this.eventTaskScheduler.schedule(
            snippet,
            resetSyncTimeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    synchronized void cancelSyncTimer() {
        if (syncTimeoutTask != null) { // Might be null if contact was never made.
            syncTimeoutTask.cancel(true);
        }
    }

}
