package org.lsst.ccs.subsystem.shutter;

import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.motorplatform.bus.AxisStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.ControllerStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.PlatformConfig;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * The publisher of all messages going out on the CCS status bus. Has a task that
 * sends out the latest status infor at regular intervals. There are also a few
 * methods for publishing some status information without delay. Thread-safe.
 * @author tether
 */
public class Publisher implements HasLifecycle {
    private static final Logger LOG = Logger.getLogger(Publisher.class.getName());

    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Subsystem subsys;

    @ConfigurationParameter(isFinal=true, description="The name of the subsystem configuration in effect.")
    private volatile String configurationName = "<No config>";

    @ConfigurationParameter(description="The interval between regular status publications.")
    private volatile Duration publicationInterval = Duration.ofSeconds(1);

    // GUARDED by the instance lock.
    // Invariants: The individual status objects for the axes
    // are derived from the same ShutterStatusPLC message. Yhe controller status
    // is derived from that and the individual status-setting methods.
    private ShutterStatus shutterStat;
    private ControllerStatus controllerStat;
    private Map<Axis, AxisStatus> axisStat;
    private PlatformConfig config;
    private Scheduler publicationScheduler;
    private PeriodicTask publicationTask;
    // END GUARDED

    /**
     * Status of command execution.
     */
    static enum CommandStatus {
        /** Unknown due to loss of contact, no command being sent yet, etc.*/
        UNKNOWN,
        /** Sent to the shutter controller. */
        SENT,
        /** The controller has sent back an acknowledgment. */
        ACKED,
        /** The command either never reached to controller or failed in some way. */
        FAILED;
    }

    /** Status of communications with the shutter controller. */
    static enum ContactStatus {
        /** Contact has been established. */
        UP,
        /** Contact is broken.*/
        DOWN;
    }

    /**
     * Establishes reasonable default values for the private fields, where feasible.
     */
    @Override
    public void init() {
        final List<String> axisNames =
            Stream.of(Axis.values()).map(Axis::getName).collect(Collectors.toList());
        final List<String> axisUnits =
            Stream.of(Axis.values()).map((x) -> "mm").collect(Collectors.toList());
        final List<String> capture = Collections.emptyList();
        final List<String> digitalInputNames = Collections.emptyList();
        final List<String> digitalOutputNames = Collections.emptyList();
        final List<String> analogInputDescriptions = Collections.emptyList();
        config = new PlatformConfig(
            axisNames, capture, digitalInputNames, digitalOutputNames,
            analogInputDescriptions, configurationName, axisUnits
        );
        axisStat = new TreeMap<>();
        controllerStat =
            new ControllerStatus(ContactStatus.DOWN.toString(), false, CommandStatus.UNKNOWN.toString());
    }

    /** Takes configuration changes into account and starts the publication task. */
    @Override
    public synchronized void postStart() {
        publicationScheduler = new Scheduler("Status publication", 1);
        publicationScheduler.setLogger(LOG);
        publicationScheduler.setDefaultLogLevel(Level.SEVERE);
        LOG.info("Starting the publication task.");
        publicationTask = publicationScheduler.scheduleWithFixedDelay(
            this::publicationTaskBody,
            0, publicationInterval.toMillis(), TimeUnit.MILLISECONDS);
        config = new PlatformConfig(
            config.getAxisNames(),
            config.getCapture(),
            config.getDigitalInputNames(),
            config.getDigitalOutputNames(),
            config.getAnalogInputDescriptions(),
            configurationName,
            config.getAxisUnits()
        );
        // The configuration name may have changed, push out the new PlatformConfig just in
        // case somebody managed to ask for the configuration before now.
        publishConfiguration();
    }

    /**
     * Shuts down the publication task and its scheduler.
     */
    @Override
    public synchronized void shutdown() {
        LOG.info("Normal stop of the publication task.");
        publicationScheduler.shutdownNow();
    }

    /**
     * Saves the shutter status received from the controller and make motorplatform
     * status messages from it.
     * @param shutterStat The latest status message received from the controller.
     */
    synchronized void updateShutterStatus(final ShutterStatus shutterStat) {
        this.shutterStat = shutterStat;
    }

    /**
     * Gets the last published shutter status.
     * @return The status value, or null if none has yet been received from the shutter controller.
     */
    synchronized ShutterStatus getShutterStatus() {
        return shutterStat;
    }

    /**
     * Retrieves the latest status for the given axis,
     * @param axisName The name of the axis.
     * @return The corresponding {@code AxisStatus} value, or null if there is none.
     */
    synchronized AxisStatus getAxisStatus(final String axisName) {
        assert axisName != null;
        final Axis ax = Axis.fromName(axisName);
        assert ax != null;
        return axisStat.get(ax);
    }

    /**
     * Sets the status of communications with the controller for later publication.
     * @param stat The new contact status.
     */
    synchronized void setContactStatus(final ContactStatus stat) {
        assert stat != null;
        controllerStat = new ControllerStatus(
            stat.toString(),
            controllerStat.isMotionEnabled(),
            stat == ContactStatus.UP ? controllerStat.getLastCommandStatus() : CommandStatus.UNKNOWN.toString()
        );
    }

    /**
     * Sets the controller enabled/disabled flag for later publication.
     * @param motionEnabled The new setting.
     */
    synchronized void setEnableStatus(final boolean motionEnabled) {
        controllerStat = new ControllerStatus(
            controllerStat.getCommLinkStatus(),
            motionEnabled,
            controllerStat.getLastCommandStatus()
        );
    }

    /**
     * Sets the status of the latest command sent to the controller, for later publication.
     * @param stat The new command status.
     */
    synchronized void setCommandStatus(final CommandStatus stat) {
        assert stat != null;
        controllerStat = new ControllerStatus(
            controllerStat.getCommLinkStatus(),
            controllerStat.isMotionEnabled(),
            stat.toString()
        );
    }

    /**
     * Publishes the entire set of status objects at regular intervals.
     */
    private void publicationTaskBody() {
        // Local copies of status references.
        ShutterStatus shutter;
        ControllerStatus controller;
        List<AxisStatus> axis;

        // Update status values while holding the instance lock.
        synchronized (this) {
            if (shutterStat == null) return; // Wait for the next update.

            // Update the contact status in the controller status.
            controllerStat = new ControllerStatus(ContactStatus.UP.toString(),
                controllerStat.isMotionEnabled(),
                controllerStat.getLastCommandStatus()
            );
            // Update the axis status values.
            for (final Axis ax: Axis.values()) {
                final ShutterStatus.AxisStatus plcStat =
                    shutterStat.getAxisStatus(ShutterSide.fromAxis(ax));
                final AxisStatus mpStat =
                    new AxisStatus(
                        ax.getName(),
                        plcStat.isEnabled(),
                        plcStat.getActVel() != 0,
                        plcStat.isHomed(),
                        plcStat.atLowLimit(),
                        plcStat.atHighLimit(),
                        plcStat.getErrorID() == 0
                            ? Collections.emptyList()
                            : Arrays.asList("Error " + plcStat.getErrorID()),
                        plcStat.getActPos()
                    );
                axisStat.put(ax, mpStat);
            }

            // Make local copies of the references.
            shutter = this.shutterStat;
            controller = this.controllerStat;
            axis = new ArrayList<>(this.axisStat.values());
        }

        // Now publish without holding the lock as network operations
        // take a long time.
        publish(shutter);
        publish(controller);
        for (final AxisStatus ax: axis) {publish(ax);}
    }

    /**
     * Immediately publishes any non-null serializable object as a {@code KeyValueData} instance,
     * @param msg If null, we do nothing. Otherwise this object is packaged for transmission
     * using the simple name of its class as the key.
     */
    public void publish(final Serializable msg) {
        if (msg != null) {
            subsys.publishSubsystemDataOnStatusBus(
                new KeyValueData(
                    msg.getClass().getSimpleName(),
                    msg
                )
            );
        }
    }

    /**
     * Immediately publishes the current controller status.
     */
    void publishControllerStatus() {
        publish(getControllerStatus());
    }

    private synchronized ControllerStatus getControllerStatus() {return controllerStat;}

    /**
     * Immediately publishes the current motorplatform configuration.
     */
    void publishConfiguration() {
        publish(getConfiguration());
    }

    private synchronized PlatformConfig getConfiguration() {return config;}

    /**
     * Immediately publishes the status of the given axis.
     * @param axisName The name of the axis.
     */
    void publishAxisStatus(final String axisName) {
        final AxisStatus stat = getAxisStatus(axisName);
        publish(stat);
    }

    /**
     * Publish a {@code MotionDone} message.
     * @param motion The motion message.
     */
    public void publishMotionDone(final MotionDone motion) {
        publish(motion);
    }

    private synchronized void setShutterStatus(final ShutterStatus shutter) {shutterStat = shutter;}

}
