package org.lsst.ccs.subsystem.shutter;

import java.time.Duration;
import java.util.EnumMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.stream.Collectors;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.common.Constants;
import org.lsst.ccs.subsystem.shutter.common.EncoderSample;
import org.lsst.ccs.subsystem.shutter.common.HallTransition;
import org.lsst.ccs.subsystem.shutter.common.PLCError;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import org.lsst.ccs.subsystem.shutter.plc.MsgToPLC;
import org.lsst.ccs.subsystem.shutter.statemachine.Actions;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.MINUSX;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.PLUSX;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.CloseShutter;
import org.lsst.ccs.subsystem.shutter.plc.GoToProd;
import org.lsst.ccs.subsystem.shutter.plc.MoveAxisAbsolutePLC;
import org.lsst.ccs.subsystem.shutter.plc.OpenShutter;
import org.lsst.ccs.subsystem.shutter.plc.ShutterStatusPLC;
import org.lsst.ccs.subsystem.shutter.plc.TakeExposure;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus.AxisStatus;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * An implementation of {@code Actions} in which no hardware operations are performed.
 * All actions are logging at info level. Subsystem-level operations such as
 * alert handling or manipulation of the state bundle are implemented.
 * @author tether
 */
public class SimulatedActions implements Actions {
    private static final Logger LOG = Logger.getLogger(SimulatedActions.class.getName());

    private final static Alert SYNC_ALERT = new Alert("SYNC",
            "Simulated status of tracking of controller state.");
    private final static Alert PLC_ALERT = new Alert("PLC",
            "Notice of simulated shutter controller errors.");

    private final Controller control;
    private final StateMachine machine;
    private final Subsystem subsys;
    private final Publisher publish;



    /**
     * Saves references to the components that actually implement the actions.
     * @param control A reference to the {@code Controller} component.
     * @param machine A reference to the {@code StateMachine} component.
     * @param subsys A reference to the subsystem object.
     * @param publish A reference to the {@code Publisher} component.
     */
    public SimulatedActions(
        final Controller control,
        final StateMachine machine,
        final Subsystem subsys,
        final Publisher publish)
    {
        this.control = control;
        this.machine = machine;
        this.subsys = subsys;
        this.publish = publish;
    }

    @Override
    public void cancelSyncTimer() {
        LOG.info("{ACTION cancelSyncTimer()}");
    }

    @Override
    public boolean isBadExposureTime(final Duration exposureTime) {
        LOG.info("{ACTION isBadExposure()}");
        return false;
    }

    @Override
    public void lowerPLCAlert() {
        LOG.info("{ACTION lowerPLCAlert()}");
        subsys.getAgentService(AlertService.class)
                .raiseAlert(PLC_ALERT, AlertState.NOMINAL, "PLC reset.");
    }

    @Override
    public void lowerSyncAlert() {
        LOG.info("{ACTION lowerSyncAlert()}");
        subsys.getAgentService(AlertService.class)
                .raiseAlert(SYNC_ALERT, AlertState.NOMINAL, "Synchronized with PLC.");
    }

    @Override
    public boolean makePartialContact() {
        LOG.info("{ACTION makePartialContact()}");
        control.simulateContact();
        return true;
    }

    @Override
    public boolean makeFullContact() {
        LOG.info("{ACTION makeFullContact()}");
        return true;
    }

    @Override
    public void publishEnableStatus(final boolean isEnabled) {
        publish.setEnableStatus(isEnabled);
    }

    @Override
    public void setPhysicalState(final PhysicalState newState) {
        LOG.info(String.format("{ACTION setPhysicalState(%s)}", newState));
        machine.setPhysicalState(newState);
    }

    @Override
    public void setSoftwareState(SoftwareState newState) {
        LOG.info(String.format("{ACTION setSoftwareState(%s)}", newState));
        machine.setSoftwareState(newState);
    }

    @Override
    public void raisePLCAlert(final org.lsst.ccs.subsystem.shutter.plc.Error err) {
        LOG.fine("{ACTION raisePLCAlert()}");
        for (final PLCError e: err.getErrors()) {
            LOG.log(Level.WARNING,
                    "Simulated PLC error {0} - {1}", new Object[]{e.toString(), e.getDescription()});
        }
        final String errlist =
            err.getErrors().stream().map(PLCError::toString).collect(Collectors.joining(", "));
        subsys.getAgentService(AlertService.class).raiseAlert(PLC_ALERT, AlertState.ALARM, errlist);
    }

    @Override
    public void raiseSyncAlert() {
        LOG.info("{ACTION raiseSyncAlert()}");
        subsys.getAgentService(AlertService.class).raiseAlert(SYNC_ALERT, AlertState.ALARM, "Lost track.");
    }

    @Override
    public boolean readyForCalibration() {
        LOG.info("{ACTION readyForCalibration()}");
        return true;
    }

    @Override
    public void resetPLC() {
        LOG.info("{ACTION resetPLC()}");
        control.simulateResetPLC();
    }

    @Override
    public void relay(final MsgToPLC eventMsg) {
        LOG.log(Level.INFO, "'{'ACTION relay({0})'}'", eventMsg.getClass());
        control.simulateRelay(eventMsg);
    }

    @Override
    public void saveCalib(final CalibDone cal) {
        LOG.info("{ACTION saveCalib() - not yet implemented.}");
    }

    @Override
    public void sendProfile(final MotionDonePLC mot) {
        LOG.info("{ACTION sendProfile()}");
        dumpMotionDone(mot.getStatusBusMessage());
        publish.publishMotionDone(mot.getStatusBusMessage());
    }

    private void dumpMotionDone(final MotionDone mot) {
        final StringBuilder bld = new StringBuilder();
        bld.append("[MotionDone begin]\n");
        // Header.
        bld.append("side " + mot.side() + "\n");
        bld.append("startPosition " + mot.startPosition() + " mm\n");
        bld.append("startTime " + mot.startTime().getUTCInstant().toEpochMilli() + " Unix epoch ms\n");
        bld.append("targetPosition " + mot.targetPosition() + " mm\n");
        bld.append("targetDuration " + mot.targetDuration().toMillis() + " ms\n");
        bld.append("endPosition " + mot.endPosition() + " mm\n");
        bld.append("actualDuration " + mot.actualDuration().toMillis() + " ms\n");
        // Hall transitions.
        bld.append("[Hall begin] time(Unix epoch ms) ID position(mm) isOn\n");
        for (final HallTransition hall: mot.hallTransitions()) {
            bld.append(hall.getTime().getUTCInstant().toEpochMilli());
            bld.append(" ");
            bld.append(hall.getSensorId());
            bld.append(" ");
            bld.append(hall.getPosition());
            bld.append(" ");
            bld.append(hall.isOn());
            bld.append("\n");
        }
        bld.append("[Hall end]\n");
        // Encoder samples.
        bld.append("[Encoder begin] time(Unix epoch ms) position(mm)\n");
        for (final EncoderSample enc: mot.encoderSamples()) {
            bld.append(enc.getTime().getUTCInstant().toEpochMilli());
            bld.append(" ");
            bld.append(enc.getPosition());
            bld.append("\n");
        }
        bld.append("[Encoder end]\n");
        bld.append("[MotionDone end]\n");
        LOG.fine(bld.toString());
    }

    @Override
    public void startFirstCentering() {
        LOG.info("{ACTION startFirstCentering()}");
        control.startFirstCentering();
    }

    @Override
    public void startSecondCentering() {
        LOG.info("{ACTION startSecondCentering(};}");
        control.startSecondCentering();
    }

    @Override
    public boolean shutterIsReady() {
        LOG.info("{ACTION shutterIsReady()}");
        return true;
    }

    @Override
    public void startSyncTimer() {
        LOG.info("{ACTION startSyncTimer()}");
    }

    @Override
    public void terminateContact() {
        LOG.info("{ACTION terminateContact()}");
    }

}
