package org.lsst.ccs.subsystem.shutter;

import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.ads.ADSDriver;
import org.lsst.ccs.drivers.ads.Notification;
import org.lsst.ccs.drivers.ads.VariableHandle;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.shutter.PLCVariableDictionary.InVariable;
import org.lsst.ccs.subsystem.shutter.PLCVariableDictionary.OutVariable;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import static org.lsst.ccs.subsystem.shutter.common.Axis.AXIS0;
import static org.lsst.ccs.subsystem.shutter.common.Axis.AXIS1;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.MINUSX;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.PLUSX;
import org.lsst.ccs.subsystem.shutter.plc.BladeSetPosition;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.Calibrate;
import org.lsst.ccs.subsystem.shutter.plc.ChangeAxisEnablePLC;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.ClearAllFaultsPLC;
import org.lsst.ccs.subsystem.shutter.plc.ClearAxisFaultsPLC;
import org.lsst.ccs.subsystem.shutter.plc.CloseShutter;
import org.lsst.ccs.subsystem.shutter.plc.Disable;
import org.lsst.ccs.subsystem.shutter.plc.DisableAllAxesPLC;
import org.lsst.ccs.subsystem.shutter.plc.Enable;
import org.lsst.ccs.subsystem.shutter.plc.EnableAllAxesPLC;
import org.lsst.ccs.subsystem.shutter.plc.Error;
import org.lsst.ccs.subsystem.shutter.plc.GoToProd;
import org.lsst.ccs.subsystem.shutter.plc.HomeAxisPLC;
import org.lsst.ccs.subsystem.shutter.plc.Ignored;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import org.lsst.ccs.subsystem.shutter.plc.MoveAxisAbsolutePLC;
import org.lsst.ccs.subsystem.shutter.plc.MoveAxisRelativePLC;
import org.lsst.ccs.subsystem.shutter.plc.MsgToCCS;
import org.lsst.ccs.subsystem.shutter.plc.MsgToPLC;
import org.lsst.ccs.subsystem.shutter.plc.OpenShutter;
import org.lsst.ccs.subsystem.shutter.plc.PLCMsg;
import org.lsst.ccs.subsystem.shutter.plc.Reset;
import org.lsst.ccs.subsystem.shutter.plc.ShutterStatusPLC;
import org.lsst.ccs.subsystem.shutter.plc.TakeExposure;
import org.lsst.ccs.subsystem.shutter.plc.Timer;
import org.lsst.ccs.subsystem.shutter.statemachine.BlackHoleChannel;
import org.lsst.ccs.subsystem.shutter.statemachine.Channel;
import org.lsst.ccs.subsystem.shutter.statemachine.EventReply;
import org.lsst.ccs.subsystem.shutter.statemachine.SynchronousChannel;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;

/**
 * The subsystem component responsible for all communication with the Beckhoff PLC controller running the
 * shutter, which includes caching the last status message(s) received from the PLC.
 *
 * @author tether
 *
 */
public class Controller {

    @ConfigurationParameter(isFinal=true, description="IP address of the shutter controller.")
    private volatile String plcIPaddr = "0.0.0.0";

    @ConfigurationParameter(isFinal=true, description="AMS address of the shutter controller.")
    private volatile String plcAMSaddr = "0.0.0.0.1.1";

    @ConfigurationParameter(isFinal=true, description="Local AMS address used by the subsystem.")
    private volatile String localAMSaddr = "0.0.0.1.1.1";

    @ConfigurationParameter(description="Startup delay for the notification reader.")
    private volatile Duration noticeReaderStartupDelay = Duration.ofSeconds(2);

    @ConfigurationParameter(description="Restart delay for the notification reader.")
    private volatile Duration noticeReaderRestartDelay = Duration.ofMillis(10);

    @ConfigurationParameter(description="Ack timeout for commands sent to the shutter controller.")
    private volatile Duration plcAckTimeout = Duration.ofSeconds(30);

    @ConfigurationParameter(description="The minimum exposure time.")
    private volatile Duration minExposureTime = Duration.ofMillis(100);

    @ConfigurationParameter(isFinal=true, description="Reference positions for the -X blade set.")
    private volatile Map<String, Double> referenceMinusXpositions = new HashMap<>();

    @ConfigurationParameter(isFinal=true, description="Reference positions for the +X blade set.")
    private volatile Map<String, Double> referencePlusXpositions = new HashMap<>();

    @ConfigurationParameter(description="Move blade sets at this speed (mm/sec) when centering. mm/sec.")
    private volatile double centeringSpeed = 20.0;
    
    @ConfigurationParameter(description="Max speed (mm/sec) to attain during a stroke.")
    private volatile double maxStrokeSpeed = 1667.0;

    private static final Logger LOG = Logger.getLogger(Controller.class.getName());

    // GUARDED by the instance lock.
    private ADSDriver driver;
    private PeriodicTask plcReceptionTask;
    private Map<Class<? extends MsgToPLC>, VariableHandle> writableVarHandles;
    private CountDownLatch plcReceptionStopLatch;
    private List<Axis> centeringOrder;
    private SimulatedShutter simShutter;
    // END GUARDED

    // GUARDED by the constructor.
    private final PLCVariableDictionary plcVarDict;
    private final AtomicReference<MessageWithState> pendingMessage;
    private final Channel<EventReply> smRecvReplyChan; // For replies StateMachine -> notice receiver.
    // END GUARDED

    @LookupField(strategy = LookupField.Strategy.TREE)
    private volatile Publisher publish;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private volatile StateMachine machine;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private volatile Subsystem subsys;

    public Controller() {
        this.referencePlusXpositions.put(BladeSetPosition.HOME.getKey(), 0.0);
        this.referencePlusXpositions.put(BladeSetPosition.RETRACTED.getKey(), 1.0);
        this.referencePlusXpositions.put(BladeSetPosition.CENTERED.getKey(), 374.5);
        this.referencePlusXpositions.put(BladeSetPosition.EXTENDED.getKey(), 749.0);
        this.referenceMinusXpositions.put(BladeSetPosition.HOME.getKey(), 750.0);
        this.referenceMinusXpositions.put(BladeSetPosition.RETRACTED.getKey(), 749.0);
        this.referenceMinusXpositions.put(BladeSetPosition.CENTERED.getKey(), 375.5);
        this.referenceMinusXpositions.put(BladeSetPosition.EXTENDED.getKey(), 1.0);

        this.smRecvReplyChan = new SynchronousChannel<>();

        final PLCVariableDictionary dict = new PLCVariableDictionary();
        // Create dictionary entries for variables used to send messages to the PLC task.
        dict.addMsgToPLC(Calibrate.class, Calibrate::new);
        dict.addMsgToPLC(ChangeAxisEnablePLC.class, ChangeAxisEnablePLC::new);
        dict.addMsgToPLC(ChangeBrakeState.class, ChangeBrakeState::new);
        dict.addMsgToPLC(ClearAllFaultsPLC.class, ClearAllFaultsPLC::new);
        dict.addMsgToPLC(ClearAxisFaultsPLC.class, ClearAxisFaultsPLC::new);
        dict.addMsgToPLC(CloseShutter.class, CloseShutter::new);
        dict.addMsgToPLC(DisableAllAxesPLC.class, DisableAllAxesPLC::new);
        dict.addMsgToPLC(EnableAllAxesPLC.class, EnableAllAxesPLC::new);
        dict.addMsgToPLC(GoToProd.class, GoToProd::new);
        dict.addMsgToPLC(MoveAxisAbsolutePLC.class, MoveAxisAbsolutePLC::new);
        dict.addMsgToPLC(MoveAxisRelativePLC.class, MoveAxisRelativePLC::new);
        dict.addMsgToPLC(HomeAxisPLC.class, HomeAxisPLC::new);
        dict.addMsgToPLC(OpenShutter.class, OpenShutter::new);
        dict.addMsgToPLC(Reset.class, Reset::new);
        dict.addMsgToPLC(TakeExposure.class, TakeExposure::new);

        // Now for the variables used to send non-ack messages to CCS.
        dict.addMsgToCCS(
            CalibDone.class,
            CalibDone::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> machine.calibDone(chan, (CalibDone)msg)
        );
        dict.addMsgToCCS(
            Disable.class,
            Disable::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> machine.disable(chan)
        );
        dict.addMsgToCCS(
            Enable.class,
            Enable::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> machine.enable(chan)
        );
        dict.addMsgToCCS(
            Error.class,
            Error::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> machine.error(chan, (Error)msg)
        );
        dict.addMsgToCCS(
            Ignored.class,
            Ignored::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> machine.ignored(chan, ((Ignored)msg).getReason())
        );
        dict.addMsgToCCS(MotionDonePLC.class,
            MotionDonePLC::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> machine.motionDone(chan, (MotionDonePLC)msg)
        );
        dict.addMsgToCCS(ShutterStatusPLC.class,
            ShutterStatusPLC::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> {}  // Not an event message.
        );
        dict.addMsgToCCS(
            Timer.class,
            Timer::new,
            (Channel<EventReply> chan, MsgToCCS msg) -> machine.timer(chan)
        );
        this.plcVarDict = dict;
        this.pendingMessage = new AtomicReference<>();
    }

    /**
     * Determines if an exposure time is too small to be valid.
     * @param exposureTime The exposure time to be tested.
     * @return false if the exposure time is OK, else true.
     */
    boolean isBadExposureTime(final Duration exposureTime) {
        return exposureTime.compareTo(minExposureTime) < 0;
    }

    /**
     * Stops the notice-reception task and closes the connection to the shutter controller.
     */
    synchronized void terminateContact() {
        if (plcReceptionTask != null) {
            plcReceptionTask.cancel(true);
        }
        if (driver != null) {
            driver.close();
        }
    }

    /**
     * Makes a new connection to the shutter controller and starts the notice-reception task, but enables only
     * the sending of the reset command and the reception of only the reset ack and the shutter status
     * message.
     * @return true if the operation succeeded, else false.
     */
    synchronized boolean makePartialContact() {
        try {
            // 1. Stop any running notification reception task.
            if (plcReceptionTask != null) {
                plcReceptionTask.cancel(true);
            }
            // 2. Create a driver if needed.
            if (driver == null) {
                driver = new ADSDriver(localAMSaddr);
            } // 3. Close any open connection. This invalidates all extant handles
            // and cancels all notifications in effect.
            else {
                driver.close();
            }
            // 4. Forget any variable handles we've been keeping.
            if (writableVarHandles != null) {
                writableVarHandles.clear();
            }
            else {
                writableVarHandles = new HashMap<>();
            }
            // 5. Open a new connection.
            driver.open(plcAMSaddr, plcIPaddr);
            // 6. Save the variable handle for the reset() message.
            final InVariable vreset = plcVarDict.getInVariable(Reset.class);
            this.writableVarHandles.put(vreset.klass, driver.getVariableHandle(vreset.varName));
            // 7. Set up notification for reset ack variable and for the
            // status message variable.
            driver.requestNotifications(
                    driver.getVariableHandle(vreset.ackName),
                    Duration.ZERO,   // Send immediately.
                    Duration.ZERO,   // Check every PLC task cycle.
                    true);           // Send only if value has changed.
            driver.requestNotifications(driver.getVariableHandle(plcVarDict.getOutVariable(ShutterStatusPLC.class).varName),
                    Duration.ZERO, Duration.ZERO, true);
            // 8. Start the task that reads notificaions.
            subsys.getScheduler().setLogger(LOG);
            startPLCReceptionTask();
            return true;
        } catch (Exception exc) {
            LOG.severe("Could not contact the shutter controller.", exc);
            try {
                machine.contactLost(smRecvReplyChan);
            } catch (InterruptedException ex) {
                java.util.logging.Logger.getLogger(Controller.class.getName()).log(Level.SEVERE, null, ex);
            }
            return false;
        }
    }

    /**
     * Enable the transmission or reception, as appropriate, for all messages.
     * @return true if the operation succeeded, else false.
     */
    synchronized boolean makeFullContact() {
        try {
            // 1. Stop the processing of received messages.
            plcReceptionTask.cancel(true /* Interrupt it. */);
            plcReceptionStopLatch.await();
            // The task MUST have stopped before we proceed since the making
            // of new notification requests will generate messages that we don't want processed.
            // 2. Store handles for all "in" variables.
            final List<InVariable> inVars = plcVarDict.getAllInVariables();
            for (final InVariable var: inVars) {
                writableVarHandles.put(var.klass, driver.getVariableHandle(var.varName));
            }
            // 3. Set up notification for all "ack" variables.
            for (final InVariable var: inVars)  {
                driver.requestNotifications(
                    driver.getVariableHandle(var.ackName),
                    Duration.ZERO, Duration.ZERO, true);
            }
            // 4. Set up notification for all "out" variables.
            final List<OutVariable> outVars = plcVarDict.getAllOutVariables();
            for (final OutVariable var: outVars) {
                driver.requestNotifications(
                    driver.getVariableHandle(var.varName),
                    Duration.ZERO, Duration.ZERO, true);
            }
            // 5. Re-start the processing of received messages.
            startPLCReceptionTask();
            return true;
        }
        catch (Exception exc) {
            LOG.severe("Lost contact with the shutter controller.", exc);
            submitContactLost();
            return false;
        }
    }

    /**
     * Determines whether the shutter is ready to go from Main mode to Prod mode. The determination
     * is based on the latest {@code ShutterStatusPLC} message received from the shutter PLC. The following
     * conditions must hold true for the shutter to be considered ready to make the transition:
     * <ul>
     * <li>A shutter status message must have been received.
     * <li>Both brakes must be released.</li>
     * <li>Both axes have been homed.</li>
     * <li>Both axes are enabled.</li>
     * <li>The shutter must be closed: One blade set is fully retracted and the other is fully extended.</li>
     * </ul>
     * @return true if all the conditions are met, else false.
     */
    synchronized boolean shutterIsReady() {
        final ShutterStatus status = publish.getShutterStatus();
        if (status == null) {
            LOG.warning("No shutter status has yet been received.");
            return false;
        }
        final ShutterInfo info = new ShutterInfo(status, referenceMinusXpositions, referencePlusXpositions);
        return info.shutterIsReady(); // Also does logging.
    }

    /**
     * Used to submit a {@code contactLost()} event to the state machine. Code being run in the state machine
     * task needs to use this method in order to prevent a possible self-sending deadlock. Other code
     * may use it as well. We start special task which submits the event, waits for the reply and
     * checks the reply.
     */
    private void submitContactLost() {
        final Runnable body = () -> {
            try {
                final Channel<EventReply> chan = new SynchronousChannel<>();
                machine.contactLost(chan);
                final EventReply reply = chan.read();
                if (!reply.wasAccepted(null)) {
                    LOG.severe("A contactLost() event was rejected by the state machine.");
                    LOG.severe(reply.getMessage());
                }
            }
            catch (InterruptedException exc) {
                LOG.warning("The submission of a contactLost() event was interrupted.");
            }
            catch (TimeoutException exc) {
                // Should not be possible.
                LOG.severe("A wait for a state machine reply timed out.");
            }
        };
        subsys.getScheduler().schedule(body, 0, TimeUnit.MILLISECONDS);
    }

    /**
     * Determines whether the shutter is ready to perform a calibration. The determination
     * is based on the latest {@code ShutterStatusPLC} message received from the shutter PLC. The following
     * conditions must hold true for the shutter to be considered ready to make the transition:
     * <ul>
     * <li>A shutter status message must have been received.
     * <li>Both brakes must be released.</li>
     * <li>Both axes have been homed.</li>
     * <li>Both axes are enabled.</li>
     * </ul>
     * @return true if all the conditions are met, else false.
     */
    synchronized boolean readyForCalibration() {
        final ShutterStatus status = publish.getShutterStatus();
        if (status == null) {
            LOG.warning("No shutter status has yet been received.");
            return false;
        }
        final ShutterInfo info = new ShutterInfo(status, referenceMinusXpositions, referencePlusXpositions);
        return info.readyForCalibration();  // Also does logging.
    }
    /**
     * Decides in which order to kove the blade sets, most extended first, then starts
     * the first motion.
     */
    synchronized void startFirstCentering() {
        final ShutterStatus status = publish.getShutterStatus();
        final ShutterInfo info = new ShutterInfo(status, referenceMinusXpositions, referencePlusXpositions);
        assert status != null : "Null shutter status";
        assert centeringOrder == null : "A centering order should not exist yet.";
        centeringOrder = info.getCenteringOrder();
        machine.getActions().relay(
            new MoveAxisAbsolutePLC(
                new MoveAxisAbsolute(
                    centeringOrder.get(0).getName(),
                    info.getCentered(centeringOrder.get(0)),
                    centeringSpeed
                )
            )
        );
    }

    /**
     * Starts the second and last blade set centering motion. {@code startFirstCentering()}
     * should have been called first.
     */
    synchronized void startSecondCentering() {
        assert centeringOrder != null : "A centering order wasn't chosen.";
        final ShutterStatus status = publish.getShutterStatus();
        final ShutterInfo info = new ShutterInfo(status, referenceMinusXpositions, referencePlusXpositions);
        machine.getActions().relay(
            new MoveAxisAbsolutePLC(
                new MoveAxisAbsolute(
                    centeringOrder.get(1).getName(),
                    info.getCentered(centeringOrder.get(1)),
                    centeringSpeed
                )
            )
        );
        centeringOrder = null;
    }


    private void startPLCReceptionTask() throws InterruptedException {
        final ADSDriver drv = driver;
        final CountDownLatch startLatch = new CountDownLatch(1);
        final CountDownLatch stopLatch = new CountDownLatch(1);
        final Runnable reception = () ->
            plcReceptionTaskBody(drv, plcVarDict, pendingMessage, publish, machine, startLatch, stopLatch);
        plcReceptionStopLatch = stopLatch;
        plcReceptionTask
                = subsys
                .getScheduler()
                .scheduleWithFixedDelay(reception,
                        noticeReaderStartupDelay.toMillis(),
                        noticeReaderRestartDelay.toMillis(),
                        TimeUnit.MILLISECONDS,
                        "Notification reception",
                        Level.SEVERE);
        startLatch.await();
    }

    ////////// PLC reception tasks //////////

    // PLC task state numbers appearing in the status messages from the controller.
    private static final int PLC_DISABLED_STATE = 10000;
    private static final int PLC_STILL_STATE = 21010;

    private void plcReceptionTaskBody
    (
        final ADSDriver driver,
        final PLCVariableDictionary plcVarDict,
        final AtomicReference<MessageWithState> pendingMessage,
        final Publisher publish,
        final StateMachine machine,
        final CountDownLatch startLatch,
        final CountDownLatch stopLatch
    )
    {
        // Clear the notice queue.
        final List<Notification> leftovers = new ArrayList<>();
        driver.drainNotifications(leftovers);
        // Allow the main thread to proceed.
        startLatch.countDown();
        // Start reading notices from the queue.
        LOG.info("The notice-reading loop has started.");
        try {
            while (true) {
                // Take a new notice from the queue. Terminate loop if interrupted.
                final Notification notice = driver.takeNotification();
                // Get the name of the variable and the message data from the notice.
                final String varName = notice.getVariableHandle().getName();
                final ByteBuffer data = notice.getData();
                processMessageData(varName, data, plcVarDict, pendingMessage, publish, machine);
            }
        }
        catch (InterruptedException exc) {
            stopLatch.countDown();
            LOG.info("Normal stop of the notice-reading loop.");
        }
    }

    private void simReceptionTaskBody
    (
        final PLCVariableDictionary plcVarDict,
        final AtomicReference<MessageWithState> pendingMessage,
        final Publisher publish,
        final StateMachine machine,
        final BlockingQueue<SimMessage> simulationQueue
    )
    {
        // Start reading notices from the queue.
        LOG.info("The simulated notice-reading loop has started.");
        try {
            while (true) {
                // Take a new notice from the queue. Terminate loop if interrupted.
                final SimMessage msg = simulationQueue.take();
                // Get the name of the variable and the message data from the notice.
                final String varName = msg.varName;
                final ByteBuffer data = msg.data;
                LOG.info("Read message with var name " + varName);
                processMessageData(varName, data, plcVarDict, pendingMessage, publish, machine);
            }
        }
        catch (InterruptedException exc) {
            LOG.info("Normal stop of the simulated notice-reading loop.");
        }
    }

    private void processMessageData(
        final String varName,
        final ByteBuffer data,
        final PLCVariableDictionary plcVarDict,
        final AtomicReference<MessageWithState> pendingMessage,
        final Publisher publish,
        final StateMachine machine) throws InterruptedException
    {
        // Check that the message-set-version is present and correct.
        // If it isn't, terminate any wait for an ack and then
        // signal loss of contact. Then skip the rest of the message processing.
        if ( checkMessageSetVersion(data, pendingMessage.get(), machine) ) {return;}
        // Perform lookups in the dictionary of PLC variables.?
        final InVariable ackVar = plcVarDict.getInVariable(varName);
        final OutVariable outVar = plcVarDict.getOutVariable(varName);
        assert (ackVar != null) || (outVar != null): "Unknown message type received from PLC.";
        assert (ackVar == null) || (outVar == null): "Ambiguous message received from PLC.";
        // If it's an acknowledegment, handle the ack.
        if (ackVar != null) {
            LOG.fine("Calling handleAckMessage().");
            handleAckMessage(data, ackVar, pendingMessage.get());
            LOG.fine("Returned from handleAckMessage()");
        }
        // If it's a status message. handle that.
        else if (outVar != null && outVar.klass == ShutterStatusPLC.class) {
            LOG.fine("Calling handleStatusMessage().");
            handleStatusMessage(data, outVar, publish, machine);
            LOG.fine("Returned from handleStatusMessage().");
        }
        // Otherwise it must be an event message, so submit it to the state machine.
        else if (outVar != null) {
            LOG.fine("Calling handleEventMessage");
            handleEventMessage(data, outVar);
            LOG.fine("Returned from handleEventMessage().");
        }
    }

    private boolean checkMessageSetVersion(
        final ByteBuffer data,
        final MessageWithState pending,
        StateMachine machine
        )
    {
        boolean failure = false;
        if (PLCMsg.messageVersionIsBad(data)) { // Will issue a log message if needed.
            if (pending != null) {
                pending.badAck();  // Will cause contactLost() to be signalled in relay().
            }
            else {
                submitContactLost();
            }
            failure = true;
        }
        return failure;
    }

    private void handleAckMessage(
        final ByteBuffer data,
        final InVariable ackVar,
        final MessageWithState pending
    )
    {
        MsgToPLC msg;
        try {
            msg = ackVar.ackDecoder.apply(data);
            if (pending != null) {
                pending.ack(msg);
            }
            else {
                LOG.severe("Unexpected ack received, message class = " + msg.getClass().getSimpleName());
            }
        }
        catch(Exception exc) {
            LOG.severe("Error during ack decoding.", exc);
        }
    }

    private void handleStatusMessage(
        final ByteBuffer data,
        final OutVariable outVar,
        final Publisher publish,
        final StateMachine machine) throws InterruptedException
    {

        ShutterStatusPLC status;
        try {
            status = (ShutterStatusPLC)outVar.decoder.apply(data);
        }
        catch (Exception exc) {
            LOG.severe("Error during status message decoding.", exc);
            return;
        }
        LOG.fine("Received ShutterStatus. " + status.toString());
        publish.updateShutterStatus(status.getStatusBusMessage());
        if (status.getSmState() == PLC_DISABLED_STATE) {
            final Channel<EventReply> chan = new BlackHoleChannel<>();  // Ignore any rejection.
            machine.plcIsDisabled(chan);
        }
        else if (status.getSmState() == PLC_STILL_STATE) {
            final Channel<EventReply> chan = new BlackHoleChannel<>();
            machine.plcIsEnabled(chan);
        }
    }

    private void handleEventMessage(final ByteBuffer data, final OutVariable outVar)
    throws InterruptedException
    {
        MsgToCCS msg;
        try {
            msg = outVar.decoder.apply(data);
        }
        catch (Exception exc) {
            LOG.severe("Error during message decoding.", exc);
            return;
        }
        outVar.submitter.submit(smRecvReplyChan, msg);
        final EventReply reply = smRecvReplyChan.read();
        try {
            if (!reply.wasAccepted(null)) {
                LOG.severe("A PLC event message was rejected by the state machine.");
                LOG.severe(reply.getMessage());
            }
        }
        catch (TimeoutException exc) {
            // Should not be possible.
            LOG.severe("A wait for a state machine reply timed out.");
        }
    }


    ////////// End of PLC reception tasks //////////

    synchronized void resetPLC() {
        relay(new Reset(maxStrokeSpeed));
    }

    void simulateResetPLC() {
        simulateRelay(new Reset(maxStrokeSpeed));
    }

    // Invariant: pendingMessage is null when this method isn't running.
    synchronized void relay(final MsgToPLC msg) {
        assert pendingMessage.get() == null: "relay() entered while still awaiting an ack.";
        try {
            pendingMessage.set(new MessageWithState(msg));
            sendPendingMessage();
            finishPendingMessage();
        }
        finally {
            pendingMessage.set(null);
        }
    }

    private void sendPendingMessage() {
        final MsgToPLC msg = pendingMessage.get().getMessage();
        try {
            final InVariable inVar = plcVarDict.getInVariable(msg.getClass());
            assert inVar != null: "Message not recognized by relay().";
            final VariableHandle handle = writableVarHandles.get(msg.getClass());
            assert handle != null: "No handle for writing the PLC message variable.";
            final ByteBuffer msgBytes = handle.createBuffer();
            msg.encode(msgBytes);
            driver.writeVariable(handle, msgBytes);
            pendingMessage.get().sent();
        }
        catch (Exception exc) {
            LOG.severe("Trouble sending a message to the PLC.", exc);
            pendingMessage.get().sendError();
        }
    }

    void simulateRelay(final MsgToPLC msg) {
        LOG.info("Simulating sending of " + msg.getClass().getSimpleName());
        assert pendingMessage.get() == null: "simulateRelay() entered while still awaiting an ack.";
        try {
            pendingMessage.set(new MessageWithState(msg));
            LOG.info("Pending message state = " + pendingMessage.get().getState().toString());
            pendingMessage.get().sent();
            simShutter.accept(msg);
            finishPendingMessage();
        }
        finally {
            pendingMessage.set(null);
        }
    }

    private BlockingQueue<SimMessage> simulationQueue;

    synchronized void simulateContact() {
        if (plcReceptionTask == null) {
            final Runnable reception = () -> {
                simReceptionTaskBody(plcVarDict, pendingMessage, publish, machine, simulationQueue);
            };
            simulationQueue = new ArrayBlockingQueue<>(1000);
            plcReceptionTask
                    = subsys
                            .getScheduler()
                            .scheduleWithFixedDelay(reception,
                                    noticeReaderStartupDelay.toMillis(),
                                    noticeReaderRestartDelay.toMillis(),
                                    TimeUnit.MILLISECONDS,
                                    "Simulated notification reception",
                                    Level.SEVERE);
            simShutter = new SimulatedShutter(subsys.getScheduler(), simulationQueue, plcVarDict);
            simShutter.start();
        }

    }

    private void finishPendingMessage() {
        LOG.fine(String.format("Waiting %s for ack.", plcAckTimeout));
        pendingMessage.get().waitForFinish(plcAckTimeout);
        LOG.fine("Done waiting.");
        final MsgToPLC msg = pendingMessage.get().getMessage();
        switch (pendingMessage.get().getState()) {
            case SENDING:
            case AWAITING_ACK:
                LOG.severe("Incorrect handling of message sending.");
                break;
            case SEND_ERROR:
                LOG.severe("Lost contact due to sending error.");
                submitContactLost();
                break;
            case ACK_TIMEOUT:
                LOG.severe("Ack timeout when sending " + msg.getClass().getSimpleName());
                submitContactLost();
                break;
            case BAD_ACK:
                LOG.severe("Mis-matched ack of " + msg.getClass().getSimpleName());
                submitContactLost();
                break;
            case GOOD_ACK:
                LOG.fine("Matching ack received.");
                break;
            case INTERRUPTED:
                LOG.severe("Interrupted while waiting for ack. This may cause trouble later.");
                break;
        }
    }

    private static class MessageWithState {
        static enum State {SENDING, AWAITING_ACK, SEND_ERROR, GOOD_ACK, BAD_ACK, ACK_TIMEOUT, INTERRUPTED};
        private final MsgToPLC msg;
        private State state;
        private MsgToPLC ackMsg;
        private final CountDownLatch latch;
        MessageWithState(final MsgToPLC msg) {
            this.msg = msg;
            this.ackMsg = null;
            this.latch = new CountDownLatch(1);
            this.state = State.SENDING;
        }
        synchronized void sent() {
            if (state == State.SENDING) {
                setState(State.AWAITING_ACK);
            }
        }
        synchronized void sendError() {
            if (state == State.SENDING) {
               setState(State.SEND_ERROR);
            }
        }
        synchronized void ack(final MsgToPLC ackMsg) {
            if (state == State.AWAITING_ACK) {
                this.ackMsg = ackMsg;
                if (ackMsg.hasSameIdent(msg)) {
                    setState(State.GOOD_ACK);
                }
                else {
                    setState(State.BAD_ACK);
                }
            }
        }
        synchronized void badAck() {
            if (state == State.AWAITING_ACK) {setState(State.BAD_ACK);} // No ack available, e.g., decoding error.
        }
        private synchronized void setState(final State newState) {
            state = newState;
            // We're done if we're no longer awaiting an ack, for whatever reason.
            if (state.compareTo(State.AWAITING_ACK) > 0) {latch.countDown();}
        }
        void waitForFinish(final Duration timeout) {
            // Must not be synchronized - can't hold lock while waiting.
            try {
                final boolean timedOut = !latch.await(timeout.toMillis(), TimeUnit.MILLISECONDS);
                if (timedOut) {setState(State.ACK_TIMEOUT);}
            }
            catch (InterruptedException exc) {
                Thread.currentThread().interrupt();
                setState(State.INTERRUPTED);
            }
        }
        MsgToPLC getMessage() {return msg;}
        synchronized MsgToPLC getAck() {return ackMsg;}
        synchronized State getState() {return state;}
    }

    // Used to answer questions about the shutter based on the latest status report and
    // some configuration parameters.
    private static class ShutterInfo {

        private final ShutterStatus status;
        private final Map<Axis, Map<BladeSetPosition, Double>> config;

        ShutterInfo(
            final ShutterStatus status,
            final Map<String, Double> minusXConfig,
            final Map<String, Double> plusXConfig)
        {
            this.status = status;
            this.config = new EnumMap<>(Axis.class);
            final Map<Axis, Map<String, Double>> conf = new EnumMap<>(Axis.class);
            conf.put(Axis.getPlusXSide(), plusXConfig);
            conf.put(Axis.getMinusXSide(), minusXConfig);
            for (final Axis ax: Axis.values()) {
                final Map<BladeSetPosition, Double> posmap = new EnumMap<>(BladeSetPosition.class);
                for (final BladeSetPosition pos: BladeSetPosition.values()) {
                    posmap.put(pos, conf.get(ax).get(pos.getKey()));
                }
                this.config.put(ax, posmap);
            }
        }

        private double getParam(final Axis axis, final BladeSetPosition pos) {return config.get(axis).get(pos);}
        public double getHome(final Axis axis) {return getParam(axis, BladeSetPosition.HOME);}
        public double getRetracted(final Axis axis) {return getParam(axis, BladeSetPosition.RETRACTED);}
        public double getCentered(final Axis axis) {return getParam(axis, BladeSetPosition.CENTERED);}
        public double getExtended(final Axis axis) {return getParam(axis, BladeSetPosition.EXTENDED);}
        public boolean isRetracted(final Axis axis) {
            final double pos = status.getAxisStatus(ShutterSide.fromAxis(axis)).getActPos();
            return Math.abs(pos - getHome(axis)) <= Math.abs(getRetracted(axis) - getHome(axis));
        }
        public boolean isExtended(final Axis axis) {
            final double pos = status.getAxisStatus(ShutterSide.fromAxis(axis)).getActPos();
            return Math.abs(pos - getHome(axis)) >= Math.abs(getExtended(axis) - getHome(axis));
        }
        public boolean shutterIsClosed() {
            return (isRetracted(AXIS0) && isExtended(AXIS1))
                   ||
                   (isExtended(AXIS0) && isRetracted(AXIS1));
        }
        public boolean axesAreEnabled() {
            return status.getAxisStatus(PLUSX).isEnabled() && status.getAxisStatus(MINUSX).isEnabled();
        }
        public boolean brakesAreReleased() {
            return !status.getAxisStatus(PLUSX).isBrakeEngaged() && !status.getAxisStatus(MINUSX).isBrakeEngaged();
        }
        public boolean axesAreHomed() {
            return status.getAxisStatus(PLUSX).isHomed() && status.getAxisStatus(MINUSX).isHomed();
        }
        public boolean shutterIsReady() {
            boolean ready = true;
            if (!shutterIsClosed()) {
                LOG.warning("Shutter not ready - not closed.");
                ready = false;
            }
            if (!axesAreEnabled()) {
                LOG.warning("Shutter not ready - axes not enabled.");
                ready = false;
            }
            if (!brakesAreReleased()) {
                LOG.warning("Shutter not ready - brakes not released.");
                ready = false;
            }
            if (!axesAreHomed()) {
                LOG.warning("Shutter not ready - axes not homed.");
                ready = false;
            }
            return ready;
        }
        public boolean readyForCalibration() {
            boolean ready = true;
            if (!axesAreEnabled()) {
                LOG.warning("Shutter not ready - axes not enabled.");
                ready = false;
            }
            if (!brakesAreReleased()) {
                LOG.warning("Shutter not ready - brakes not released.");
                ready = false;
            }
            return ready;
        }
        public List<Axis> getCenteringOrder() {
            final double extent1 =
                Math.abs(status.getAxisStatus(ShutterSide.fromAxis(AXIS0)).getActPos() - getHome(AXIS0));
            final double extent2 =
                Math.abs(status.getAxisStatus(ShutterSide.fromAxis(AXIS1)).getActPos() - getHome(AXIS1));
            if (extent1 >= extent2) {
                return Arrays.asList(AXIS0, AXIS1);
            }
            else {
                return Arrays.asList(AXIS1, AXIS0);
            }
        }
    }
}
