package org.lsst.ccs.subsystem.shutter.statemachine;

import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import java.util.logging.Logger;
import org.lsst.ccs.subsystem.shutter.plc.Reset;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEntry;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEvent;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logExit;
import static org.lsst.ccs.subsystem.shutter.statemachine.PromptReply.ACCEPTED;

/**
 * The state in which the shutter is in its normal operating mode used for opening, closing and
 * taking exposures. Thread-safe.
 * @author tether
 */
class Prod extends CompositeState<Prod, Enabled> {
    private static final Logger LOG = Logger.getLogger(Prod.class.getName());

    // PROTECTED by the instance lock.
    private Closed closedState;
    private Opening openingState;
    private Traveling travelingState;
    private Opened openedState;
    private Closing closingState;
    // END PROTECTED

    /** Saves the actions and context for this composite state.
     * @param actions The set of actions for this context.
     * @param context The super-context for this context.
     * @throws NullPointerException if either argument is null.
     */
    public Prod(Actions actions, Enabled context) {
        super(actions, context);
    }

    @Override
    public void init() {
        makeTransition(getClosedState(), null);
    }

    @Override
    public Logger getLogger() {
        return LOG;
    }
    
    /**
     * Gets the {@code Closed} state for this context, creating it the first time through.
     * @return The {@code Closed} state.
     */
    synchronized Closed getClosedState() {
        if (closedState == null) {closedState = new Closed(this);}
        return closedState;
    }
    
    /**
     * Gets the {@code Opening} state for this context, creating it the first time through.
     * @return The {@code Opening} state.
     */
    synchronized Opening getOpeningState() {
        if (openingState == null) {openingState = new Opening(this);}
        return openingState;
    }
    
    /**
     * Gets the {@code Traveling} state for this context, creating it the first time through.
     * @return The {@code Traveling} state.
     */
    synchronized Traveling getTravelingState() {
        if (travelingState == null) {travelingState = new Traveling(this);}
        return travelingState;
    }
    
    /**
     * Gets the {@code Opened} state for this context, creating it the first time through.
     * @return The {@code Opened} state.
     */
    synchronized Opened getOpenedState() {
        if (openedState == null) {openedState = new Opened(this);}
        return openedState;
    }
    
    /**
     * Gets the {@code Closing} state for this context, creating it the first time through.
     * @return The {@code Closing} state.
     */
    synchronized Closing getClosingState() {
        if (closingState == null) {closingState = new Closing(this);}
        return closingState;
    }

    /**
     * {@inheritDoc } Handles the {@code reset()} event for all substates of this composite state.
     * @return {@link PromptReply#ACCEPTED}
     */
    @Override
    public EventReply reset() {
        final Enabled ctx = getContext();
        logEvent(this);
        ctx.makeTransition(ctx.getMaintState(),
                           () -> {ctx.getActions().setPhysicalState(PhysicalState.OTHER);
                                  ctx.getActions().resetPLC();});
        return ACCEPTED;
    }

    /**
     * {@inheritDoc} Handles the {@code error()} event for all substates of this composite state.
     * @return {@link PromptReply#ACCEPTED}
     */
    @Override
    public EventReply error() {
        final Enabled ctx = getContext();
        logEvent(this);
        ctx.makeTransition(ctx.getMaintState(),
                           () -> ctx.getActions().setPhysicalState(PhysicalState.OTHER));
        return ACCEPTED;
    }
}
