package org.lsst.ccs.subsystem.shutter;

import java.time.Duration;
import java.util.concurrent.TimeoutException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING1;
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.subsystem.motorplatform.bus.AxisStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.HomeAxis;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.motorplatform.bus.SendAxisStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.SendConfiguration;
import org.lsst.ccs.subsystem.motorplatform.bus.SendControllerStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.StopAllMotion;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.plc.Calibrate;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.statemachine.EventReply;

/**
 * Main subsystem component, accepts commands.
 *
 * @author tether
 */
public class Main {

    private final static int COMMAND_TIMEOUT = 30; // seconds
    private final static Duration RENDEZVOUS_TIMEOUT =  Duration.ofSeconds(COMMAND_TIMEOUT - 5);

    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Subsystem subsys;

    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile StateMachine machine;

    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Publisher publish;

    public Main() {}

    @Command(autoAck=false, level=NORMAL, type=ACTION, description="Try again to sync with the shutter.",
    timeout=COMMAND_TIMEOUT)
    public void resync() {
        ;
        getStateMachineReply(machine.resync());
    }
    
    private void getStateMachineReply(final EventReply reply) {
        try {
            if (reply.wasAccepted(RENDEZVOUS_TIMEOUT)) {
                subsys.sendAck(null);
            }
            else {
                subsys.sendNack(reply.getMessage());
            }
        }
        catch (TimeoutException exc) {
            subsys.sendNack("Timeout inside the subsystem.");
        }
        catch (InterruptedException exc) {
            subsys.sendNack("Command was interrupted!");
        }
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="Start a calibration.",
        timeout=COMMAND_TIMEOUT)
    public void calibrate() {
        getStateMachineReply(machine.calibrate(new Calibrate()));
    }

    @Command(autoAck=false, level=NORMAL, type=ACTION,
        description="Take an exposure for a given number of seconds.",
        timeout=COMMAND_TIMEOUT)
    public void takeExposure(final double exposureTime) throws InterruptedException {
        final long millis = Math.round( exposureTime * 1000.0);
        getStateMachineReply(machine.takeExposure(Duration.ofMillis(millis)));
    }

    @Command(autoAck=false, level=NORMAL, type=ACTION, description="Open the shutter.",
        timeout=COMMAND_TIMEOUT)
    public void openShutter() throws InterruptedException {
        getStateMachineReply(machine.openShutter());
    }

    @Command(autoAck=false, level=NORMAL, type=ACTION,
        description="Close the shutter, aborting any exposure.",
        timeout=COMMAND_TIMEOUT)
    public void closeShutter() throws InterruptedException {
        getStateMachineReply(machine.closeShutter());
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="Change the brake state on an axis.",
        timeout=COMMAND_TIMEOUT)
    public void changeBrakeState(final String axisName, final boolean brakeEngaged)
    throws InterruptedException
    {
        final Axis ax = Axis.fromName(axisName);
        if (ax == null) {
            subsys.sendNack("Invalid axis name, use '+X' or '-X'.");
        }
        getStateMachineReply(machine.changeBrakeState(ax,
                    brakeEngaged ? ChangeBrakeState.State.ENGAGED : ChangeBrakeState.State.RELEASED));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="Enter production (NORMAL) mode.",
        timeout=COMMAND_TIMEOUT)
    public void gotoProd() throws InterruptedException {
        getStateMachineReply(machine.gotoProd());
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="Center the blades (power-off prep).",
        timeout=COMMAND_TIMEOUT)
    public void gotoCenter() throws InterruptedException {
        getStateMachineReply(machine.gotoCenter());
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Start an axis homing.",
        timeout=COMMAND_TIMEOUT)
    public void homeAxis(final HomeAxis req) throws InterruptedException {
        subsys.sendNack("Axis homing is part of calibration.");
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="(Scripting) Start a move to absolute position.",
        timeout=COMMAND_TIMEOUT)
    public void moveAxisAbsolute(final MoveAxisAbsolute req) throws InterruptedException {
        getStateMachineReply(machine.moveAxisAbsolute(req));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="(Scripting) Start a position change.",
        timeout=COMMAND_TIMEOUT)
    public void moveAxisRelative(final MoveAxisRelative req) throws InterruptedException {
        getStateMachineReply(machine.moveAxisRelative(req));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="(Scripting) Clear faults on both axes.",
        timeout=COMMAND_TIMEOUT)
    public void clearAllFaults(final ClearAllFaults req) throws InterruptedException {
        getStateMachineReply(machine.clearAllFaults(req));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Enable/disable axis.",
        timeout=COMMAND_TIMEOUT)
    public void changeAxisEnable(final ChangeAxisEnable req) throws InterruptedException {
        getStateMachineReply(machine.changeAxisEnable(req));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="(Scripting) Clear faults for one axis.",
        timeout=COMMAND_TIMEOUT)
    public void clearAxisFaults(final ClearAxisFaults req) throws InterruptedException {
        getStateMachineReply(machine.clearAxisFaults(req));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Enable both axes.",
        timeout=COMMAND_TIMEOUT)
    public void enableAllAxes(final EnableAllAxes req) throws InterruptedException {
        getStateMachineReply(machine.enableAllAxes(req));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Disable both axes.",
        timeout=COMMAND_TIMEOUT)
    public void disableAllAxes(final DisableAllAxes req) throws InterruptedException {
        getStateMachineReply( machine.disableAllAxes(req));
    }

    @Command(autoAck=false, level=NORMAL, type=ACTION, description="(Scripting) Emergency stop motion.",
        timeout=COMMAND_TIMEOUT)
    public void stopAllMotion(final StopAllMotion req) throws InterruptedException {
        getStateMachineReply(machine.reset());
    }

    @Command(level=NORMAL, type=QUERY, description="(Scripting) Publish controller status ASAP.",
        timeout=COMMAND_TIMEOUT)
    public void sendControllerStatus(SendControllerStatus cmd) {
        publish.publishControllerStatus();
    }

    @Command(level=NORMAL, type=QUERY, description="(Scripting) Publish controller configuration ASAP.",
        timeout=COMMAND_TIMEOUT)
    public void sendConfiguration(SendConfiguration cmd) {
        publish.publishConfiguration();
    }

    @Command(level=NORMAL, type=QUERY, description="(Scripting) Publish axis status ASAP.",
        timeout=COMMAND_TIMEOUT)
    public void sendAxisStatus(SendAxisStatus cmd) {
        publish.publishAxisStatus(cmd.getAxisName());
    }

    ////////// Motorplatform command aliases meant for manual use from consoles. //////////
    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="(Console) Clear faults on both axes.",
        timeout=COMMAND_TIMEOUT)
    public void clearFaults() throws InterruptedException {
        clearAllFaults(new ClearAllFaults());
    }

    @Command(autoAck=false, level=NORMAL, type=ACTION, description="(Console) Emergeny stop motion.",
        timeout=COMMAND_TIMEOUT)
    public void stopAll() throws InterruptedException {
        stopAllMotion(new StopAllMotion());
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Console) Enable an axis.",
        timeout=COMMAND_TIMEOUT)
    public void enable(String axisName) throws InterruptedException {
        changeAxisEnable(new ChangeAxisEnable(axisName, true));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Console) Disable an axis.",
        timeout=COMMAND_TIMEOUT)
    public void disable(String axisName) throws InterruptedException {
        changeAxisEnable(new ChangeAxisEnable(axisName, false));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Console) Start an axis homing.",
        timeout=COMMAND_TIMEOUT)
    public void home(String axisName) throws InterruptedException {
        homeAxis(new HomeAxis(axisName));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="(Console) Start a move to an absolute position.",
        timeout=COMMAND_TIMEOUT)
    public void moveTo(String axisName, double newPositionMm, double speedMmSec) throws InterruptedException {
        moveAxisAbsolute(new MoveAxisAbsolute(axisName, newPositionMm, speedMmSec));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION,
        description="(Console) Start a position change.",
        timeout=COMMAND_TIMEOUT)
    public void moveBy(String axisName, double positionChangeMm, double seconds) throws InterruptedException {
        moveAxisRelative(new MoveAxisRelative(axisName, positionChangeMm,
            Duration.ofMillis((long)Math.rint(seconds * 1e3))));
    }

    @Command(level=NORMAL, type=QUERY, description="(Console) Show status info for an axis.",
        timeout=COMMAND_TIMEOUT)
    public String status(String axisName) {
        final StringBuilder buff = new StringBuilder(1024);
        buff.append("Axis %s%n");
        buff.append("Enabled %s, Moving %s, At Home %s, At Low %s, At High %s%n");
        buff.append("Position %6.1f%n");
        final AxisStatus axstat = publish.getAxisStatus(axisName);
        if (axstat == null) {
            return "Not available at present.";
        }
        for (String fault: axstat.getFaults()) {
            buff.append(fault);
            buff.append("%n");
        }
        return String.format(buff.toString(),
                axstat.getAxisName(),
                axstat.isEnabled(),
                axstat.isMoving(),
                axstat.isAtHome(),
                axstat.isAtLowLimit(),
                axstat.isAtHighLimit(),
                axstat.getPosition());
    }

}
