package org.lsst.ccs.subsystem.shutter.statemachine;

import java.time.Duration;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.HomeAxis;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.Calibrate;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.Ignored;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;

/**
 * Owner of a set of states and manager of transitions between them. Has a name,
 * a current state and a set of action implementations.
 * Is responsible for calling state entry methods, transition actions and
 * state exit methods in the correct order. Contexts may occur at many levels in
 * a state machine. The main context contains the state machine as a whole. A composite
 * state is a context for its substates and is a state for an owning context.
 * <p>
 * A context implements methods for all events, handling some itself and forwarding
 * the rest to the current state. This interface defines default event methods which
 * all forward to the current state.
 * <p>
 * A context provides an initialization method which makes sure that the right state
 * is the current state, if need be creating that state and making an internal transition
 * to it. This method is called whenever the context is entered. That's just once for the top-level
 * context but for a composite state it's whenever a transition occurs to it.
 * <p>
 * A context creates each state that it manages at most once. Normally this will happen on
 * the first transition to a state.
 *
 * @param <C> The concrete subclass that's extending this one. This allows state
 * fields defined here to given the type {@code State<C>} rather than just {@code State}.
 * @see CompositeState
 * @author tether
 */
interface Context<C extends Context<?>> extends Events {

    /**
     * Gets the name of this context.
     * @return the name string.
     */
    String getName();

    /**
     * Gets the current state of this context.
     * @return The current state.
     */
    State<C> getState();
    
    /**
     * Prepares the context for execution.
     * @implSpec At a minimum any implementation must make sure that the right state is current.
     */
    void init();
    
    /**
     * Cleans up a context that's not in active use. For example, resets the state to null.
     */
    void finish();
    
    /**
     * Obtains the set of action implementations owned by this context.
     * @return 
     */
    Actions getActions();
    
    /**
     * Performs a state transition, calling the appropriate exit and
     * entry methods while running the transition action in between.
     * @param newState the next value for the current state.
     * @param action The action, possibly null, associated with the transition.
     * @throws NullPointerException if {@code newState} is null.
     */
    void makeTransition(State<C> newState, Runnable action);

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply contactLost() {
        return getState().contactLost();
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply ignored(final Ignored.Reason reason) {
        return getState().ignored(reason);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply plcIsEnabled() {
        return getState().plcIsEnabled();
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply plcIsDisabled() {
        return getState().plcIsDisabled();
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply resync() {
        return getState().resync();
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply syncTimeout() {
        return getState().syncTimeout();
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply enable() {
        return getState().enable();
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply disable() {
        return getState().disable();
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply motionDone(MotionDonePLC mot) {
        return getState().motionDone(mot);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply calibrate(Calibrate cal) {
        return getState().calibrate(cal);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply calibDone(CalibDone cal) {
        return getState().calibDone(cal);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply error() {
        return getState().error();
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply reset() {
        return getState().reset();
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply takeExposure(Duration exposureTime) {
        return getState().takeExposure(exposureTime);
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply openShutter() {
        return getState().openShutter();
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply timer() {
        return getState().timer();
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply closeShutter() {
        return getState().closeShutter();
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply gotoProd() {
        return getState().gotoProd();
    }

    /**
     * @implNote This default implementation forwards to the current state.
     */
    @Override
    default EventReply gotoCenter() {
        return getState().gotoCenter();
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    default EventReply moveAxisAbsolute(MoveAxisAbsolute req) {
        return getState().moveAxisAbsolute(req);
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    default EventReply moveAxisRelative(MoveAxisRelative req) {
        return getState().moveAxisRelative(req);
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    default EventReply clearAllFaults(ClearAllFaults req) {
        return getState().clearAllFaults(req);
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    default EventReply changeAxisEnable(ChangeAxisEnable req) {
        return getState().changeAxisEnable(req);
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    public default EventReply changeBrakeState(Axis ax, ChangeBrakeState.State newState) {
        return getState().changeBrakeState(ax, newState);
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    default EventReply clearAxisFaults(ClearAxisFaults req) {
        return getState().clearAxisFaults(req);
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    default EventReply enableAllAxes(EnableAllAxes req) {
        return getState().enableAllAxes(req);
    }

    /** @implNote This default implementation forwards to the current state. */
    @Override
    default EventReply disableAllAxes(DisableAllAxes req) {
        return getState().disableAllAxes(req);
    }
}
