package org.lsst.ccs.subsystem.shutter;

import java.nio.ByteBuffer;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.shutter.PLCVariableDictionary.InVariable;
import org.lsst.ccs.subsystem.shutter.PLCVariableDictionary.OutVariable;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.common.Constants;
import org.lsst.ccs.subsystem.shutter.common.EncoderSample;
import org.lsst.ccs.subsystem.shutter.common.HallTransition;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.MINUSX;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.PLUSX;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.CloseShutter;
import org.lsst.ccs.subsystem.shutter.plc.GoToProd;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import org.lsst.ccs.subsystem.shutter.plc.MoveAxisAbsolutePLC;
import org.lsst.ccs.subsystem.shutter.plc.MsgToPLC;
import org.lsst.ccs.subsystem.shutter.plc.OpenShutter;
import org.lsst.ccs.subsystem.shutter.plc.PLCMsg;
import org.lsst.ccs.subsystem.shutter.plc.ShutterStatusPLC;
import org.lsst.ccs.subsystem.shutter.plc.TakeExposure;
import org.lsst.ccs.subsystem.shutter.sim.CubicSCurve;
import org.lsst.ccs.subsystem.shutter.sim.MotionProfile;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;
import org.lsst.ccs.utilities.scheduler.Scheduler;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Uses a thread to simulate an external shutter controller.
 * @author tether
 */
public final class SimulatedShutter {
    private static final Logger LOG = Logger.getLogger(SimulatedShutter.class.getName());

    // SHARED, must be thread-safe.
    private final Scheduler scheduler;
    private final BlockingQueue<SimMessage> outQueue;
    private final BlockingQueue<MsgToPLC> inQueue;
    private final PLCVariableDictionary dictionary;
    // END SHARED

    // LOCAL to simulation task.
    private final Map<ShutterSide, SideInfo> sdinfo;
    private ShutterSide closingSide;
    // END LOCAL

    SimulatedShutter(
        final Scheduler scheduler,
        final BlockingQueue<SimMessage> outQueue,
        final PLCVariableDictionary dictionary)
    {
        this.scheduler = scheduler;
        this.inQueue = new ArrayBlockingQueue<>(1000);
        this.outQueue = outQueue;
        this.dictionary = dictionary;
        this.sdinfo = new EnumMap<>(ShutterSide.class);
    }

    void start() {
        LOG.info("Starting controller simulation.");
        scheduler.scheduleWithFixedDelay(this::taskBody, 0, 10, TimeUnit.MILLISECONDS);
    }
    
    void accept(final MsgToPLC msg) {
        LOG.info(String.format("Accepting message %s.", msg.getClass().getSimpleName()));
        if (!inQueue.offer(msg)) {LOG.warning("Input queue is full.");}
    }

    ////////// Simulation task code //////////
    private void taskBody() {
        setInitialPositions();
        sendStatus();
        LOG.info("In event loop for simulated shutter.");
        try {
            while (true) {
                final MsgToPLC msg = inQueue.take();
                final InVariable var = dictionary.getInVariable(msg.getClass());
                if (var == null) {
                    LOG.warning(
                        String.format("Incoming message %s not in dictionary.",
                                      msg.getClass().getName()));
                }
                else {
                    sendMessage(var.ackName, msg);
                    interpretMessage(msg);
                }
            }
        }
        catch (InterruptedException exc) {
            LOG.info("Normal termination of shutter controller simulation.");
        }
    }

    private void sendMessage(final String varName, final PLCMsg msg) {
       LOG.info(String.format("Sending message %s.", msg.getClass().getSimpleName()));
       final ByteBuffer data = ByteBuffer.allocate(10240);
       msg.encode(data);
       data.limit(data.position());
       data.rewind();
       final SimMessage reply = new SimMessage(varName, data);
       if (!outQueue.offer(reply)) {LOG.warning("Output queue is full.");}
    }

    private void interpretMessage(final MsgToPLC eventMsg) {
        if (eventMsg instanceof OpenShutter) {
            final ShutterSide side = leastExtended().opposite();
            simulateMotion(side, sdinfo.get(side).homePos);
            closingSide = side.opposite();
        }
        else if (eventMsg instanceof CloseShutter) {
            final ShutterSide side = closingSide;
            simulateMotion(side, sdinfo.get(side).deployedPos);
            closingSide = null;
        }
        else if (eventMsg instanceof TakeExposure) {
            final ShutterSide side = leastExtended().opposite();
            simulateMotion(side, sdinfo.get(side).homePos);
            simulateMotion(side.opposite(), sdinfo.get(side.opposite()).deployedPos);
        }
        else if (eventMsg instanceof MoveAxisAbsolutePLC) {
            final MoveAxisAbsolute abs = ((MoveAxisAbsolutePLC)eventMsg).getStatusBusMessage();
            final ShutterSide side = ShutterSide.fromAxis(Axis.fromName(abs.getAxisName()));
            simulateMotion(side, abs.getPosition());
            closingSide = null;
        }
        else if (eventMsg instanceof GoToProd) {
            setInitialPositions();
            sendStatus();
        }
        else if (eventMsg instanceof ChangeBrakeState) {
            simulateChangeBrakeState((ChangeBrakeState)eventMsg);
        }

    }

    private void simulateChangeBrakeState(final ChangeBrakeState msg) {
        final ShutterSide side =
                msg.getAxis().isPlusXSide() ? ShutterSide.PLUSX : ShutterSide.MINUSX;
        final boolean brakeEngaged = msg.getState() == ChangeBrakeState.State.ENGAGED;
        sdinfo.get(side).brakeEngaged = brakeEngaged;
        sendStatus();

    }

    private void setInitialPositions() {
        // Shutter closed with the +X blades extended and the -X ones retracted.
        this.sdinfo.put(PLUSX,
                        new SideInfo(Constants.MAX_STROKE_LENGTH, false, 0.0, Constants.MAX_STROKE_LENGTH));
        this.sdinfo.put(MINUSX,
                        new SideInfo(Constants.MAX_STROKE_LENGTH, false, Constants.MAX_STROKE_LENGTH, 0.0));
        closingSide = null;
    }

    private ShutterSide leastExtended() {
        if (sdinfo.get(ShutterSide.PLUSX).extent() < sdinfo.get(ShutterSide.MINUSX).extent()) {
            return ShutterSide.PLUSX;
        }
        else {
            return ShutterSide.MINUSX;
        }
    }
    
    // Cover motion times on both sides of 1 second in order to test GUI graph time scaling.
    private static final long[] MOTION_TIME = new long[] {730, 1000, 2000};
    private int timeCycle = 0;

    private void simulateMotion(final ShutterSide side, final double targetPos) {
        final double startPos = sdinfo.get(side).position;
        final CCSTimeStamp startTime = CCSTimeStamp.currentTime();
        final Duration dur = Duration.ofMillis(MOTION_TIME[(timeCycle++) % MOTION_TIME.length]);
        final MotionDone.Builder motionBuilder =
            new MotionDone.Builder()
            .side(side)
            .startPosition(startPos)
            .targetPosition(targetPos)
            .startTime(startTime)
            .endPosition(targetPos)
            .targetDuration(dur)
            .actualDuration(dur)
            .hallTransitions(new ArrayList<>())
            .encoderSamples(new ArrayList<>());
        final MotionProfile profile = new CubicSCurve(targetPos - startPos, 1e-3 * dur.toMillis());
        motionBuilder.encoderSamples(makeEncoderSamples(startTime, startPos, dur, profile));
        motionBuilder.hallTransitions(makeHallTransitions(startTime, startPos, dur, profile));
        sdinfo.get(side).position = targetPos;
        final MotionDonePLC plcMotion = new MotionDonePLC(0, motionBuilder.build());
        final OutVariable var = dictionary.getOutVariable(plcMotion.getClass());
        if (var == null) {
           LOG.warning(
                String.format("Outgoing message %s not in dictionary.",
                        plcMotion.getClass().getName()));
        }
        else {
            sendMessage(var.varName, plcMotion);
        }
        
        // Update current status.
        sendStatus();
    }

    private List<EncoderSample> makeEncoderSamples(
        final CCSTimeStamp t0Stamp,
        final double startPos,
        final Duration moveTime,
        final MotionProfile profile)
    {
        final int NUM_SAMPLES = 10;
        Instant t0 = t0Stamp.getUTCInstant(); // TODO Must be UTC due to use of currentTimeFromMillis() below.
        Duration dt = moveTime.dividedBy(NUM_SAMPLES + 1);
        Instant t = t0.plus(dt);
        final List<EncoderSample> samp = new ArrayList<>(NUM_SAMPLES);
        for (int i = 0; i < NUM_SAMPLES; ++i) {
            final CCSTimeStamp stamp = CCSTimeStamp.currentTimeFromMillis(t.toEpochMilli());
            final double elapsed = 1e-3 * Duration.between(t0, stamp.getUTCInstant()).toMillis();
            samp.add(
                new EncoderSample(stamp, startPos + profile.distance(elapsed))
            );
            t = t.plus(dt);
        }
        return samp;
    }

    private List<HallTransition> makeHallTransitions(
        final CCSTimeStamp t0Stamp,
        final double startPos,
        final Duration moveTime,
        final MotionProfile profile)
    {
        final int NUM_SENSORS = 8;
        final Duration hallWidth = moveTime.dividedBy(50);
        Instant t0 = t0Stamp.getUTCInstant(); // TODO Must be UTC due to use of currentTimeFromMillis() below.
        Duration dt = moveTime.dividedBy(NUM_SENSORS + 1);
        Instant t = t0.plus(dt);
        final List<HallTransition> hall = new ArrayList<>(NUM_SENSORS);
        for (int i = 0; i < NUM_SENSORS; ++i) {
            CCSTimeStamp stamp = CCSTimeStamp.currentTimeFromMillis(t.toEpochMilli());
            double elapsed = 1e-3 * Duration.between(t0, stamp.getUTCInstant()).toMillis();
            hall.add( // off -> on
                new HallTransition(stamp, i, startPos + profile.distance(elapsed), true)
            );
            stamp = CCSTimeStamp.currentTimeFromMillis(t.plus(hallWidth).toEpochMilli());
            elapsed = 1e-3 * Duration.between(t0, stamp.getUTCInstant()).toMillis();
            hall.add( // on -> off
                new HallTransition(stamp, i, startPos + profile.distance(elapsed), false)
            );
            t = t.plus(dt);
        }
        return hall;
    }

    private void sendStatus() {
        // Make sure that clients know the current positions and other status info.
        final Map<ShutterSide, ShutterStatus.AxisStatus> axes = new EnumMap<>(ShutterSide.class);
        for (final ShutterSide side: ShutterSide.values()) {
            final ShutterStatus.AxisStatus axstat = new ShutterStatus.AxisStatus(
                sdinfo.get(side).position,
                0.0, // Velocity
                0.0, // Acceleration
                true, // Enabled?
                sdinfo.get(side).brakeEngaged,
                false, // At low limit?
                false, // At high limit?
                true, // Has been homed?
                0, // Error code.
                30.0 // Motor temp, Celsius.
            );
            axes.put(side, axstat);
        }
        ShutterStatus status = new ShutterStatus(
            0, // Motion profile code.
            true, // Shutter calibrated?
            0, // PLC state code.
            axes
        );
        final ShutterStatusPLC plcStatus = new ShutterStatusPLC(0, status);
        final OutVariable var = dictionary.getOutVariable(plcStatus.getClass());
        if (var == null) {
            LOG.warning(
                String.format("Outgoing message %s not in dictionary.",
                        plcStatus.getClass().getName()));
        }
        else {
            sendMessage(var.varName, plcStatus);
        }
    }

    private final static class SideInfo {
        double position;
        boolean brakeEngaged;
        final double homePos;
        final double deployedPos;
        final double fullStroke;
        final double direction;
        SideInfo(double position, boolean brakeEngaged, double homePos, double deployedPos) {
            this.position = position;
            this.brakeEngaged = brakeEngaged;
            this.homePos = homePos;
            this.deployedPos = deployedPos;
            final double s = deployedPos - homePos;
            this.direction = Math.signum(s);
            this.fullStroke = Math.abs(s);
        }
        boolean isExtended() {return extent() > 0.99;}
        boolean isRetracted(){return extent() < 0.01;}
        double extent() {return direction * (position - homePos) / fullStroke;}
    }

}
