package org.lsst.ccs.subsystem.shutter.statemachine;

import java.util.logging.Logger;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.plc.Calibrate;
import org.lsst.ccs.subsystem.shutter.plc.ChangeAxisEnablePLC;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.ClearAllFaultsPLC;
import org.lsst.ccs.subsystem.shutter.plc.ClearAxisFaultsPLC;
import org.lsst.ccs.subsystem.shutter.plc.DisableAllAxesPLC;
import org.lsst.ccs.subsystem.shutter.plc.EnableAllAxesPLC;
import org.lsst.ccs.subsystem.shutter.plc.MoveAxisAbsolutePLC;
import org.lsst.ccs.subsystem.shutter.plc.MoveAxisRelativePLC;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEntry;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEvent;
import static org.lsst.ccs.subsystem.shutter.statemachine.PromptReply.ACCEPTED;

/**
 * The maint-mode state in which neither blade set is moving. Thread-safe.
 * @author tether
 */
class Still extends SimpleState<Maint> {
    private static final Logger LOG = Logger.getLogger(Still.class.getName());

    /**
     * Saves the context for this state.
     * @param context The context.
     * @throws NullPointerException if the context is null.
     */
    public Still(final Maint context) {
        super(context);
    }

    @Override
    public Logger getLogger() {
        return LOG;
    }

    @Override
    public void entry() {
        logEntry(this);
        getContext().getActions().setSoftwareState(SoftwareState.STILL);
    }

    @Override
    public EventReply enableAllAxes(final EnableAllAxes req) {
        logEvent(this);
        EventReply reply = ACCEPTED;
        if (req == null) {
            reply = new PromptReply("Null argument for enableAllAxes() event.");
        }
        else {
            final Maint ctx = getContext();
            ctx.getActions().relay(new EnableAllAxesPLC(req));
        }
        return reply;
    }

    @Override
    public EventReply disableAllAxes(final DisableAllAxes req) {
        logEvent(this);
        EventReply reply = ACCEPTED;
        if (req == null) {
            reply = new PromptReply("Null argument for disableAllAxes() event.");
        }
        else {
            final Maint ctx = getContext();
            ctx.getActions().relay(new DisableAllAxesPLC(req));
        }
        return reply;
    }

    @Override
    public EventReply clearAxisFaults(final ClearAxisFaults req) {
        logEvent(this);
        EventReply reply = ACCEPTED;
        if (req == null) {
            reply = new PromptReply("Null argument for clearAxisFaults() event.");
        }
        else {
            final Maint ctx = getContext();
            ctx.getActions().relay(new ClearAxisFaultsPLC(req));
        }
        return reply;
    }

    @Override
    public EventReply changeAxisEnable(final ChangeAxisEnable req) {
        logEvent(this);
        EventReply reply = ACCEPTED;
        if (req == null) {
            reply = new PromptReply("Null argument for changeAxisEnable() event.");
        }
        else {
            final Maint ctx = getContext();
            ctx.getActions().relay(new ChangeAxisEnablePLC(req));
        }
        return reply;
    }

    @Override
    public EventReply changeBrakeState(Axis ax, ChangeBrakeState.State newState) {
        logEvent(this);
        EventReply reply = ACCEPTED;
        if (ax == null || newState == null) {
            reply = new PromptReply("The axis or the new brake state is null.");
        }
        else {
            getContext().getActions().relay(new ChangeBrakeState(ax, newState));
        }
        return reply;
    }

    @Override
    public EventReply clearAllFaults(final ClearAllFaults req) {
        logEvent(this);
        EventReply reply = ACCEPTED;
        if (req == null) {
            reply = new PromptReply("Null argument for clearAllFaults() event.");
        }
        else {
            final Maint ctx = getContext();
            ctx.getActions().relay(new ClearAllFaultsPLC(req));
        }
        return reply;
    }

    @Override
    public EventReply moveAxisRelative(final MoveAxisRelative req) {
        logEvent(this);
        EventReply reply = ACCEPTED;
        if (req == null) {
            reply = new PromptReply("Null argument for moveAxisRelative() event.");
        }
        else {
            final Maint ctx = getContext();
            ctx.makeTransition(
                ctx.getMovingState(),
                () -> ctx.getActions().relay(new MoveAxisRelativePLC(req)));
        }
        return reply;
    }

    @Override
    public EventReply moveAxisAbsolute(final MoveAxisAbsolute req) {
        logEvent(this);
        EventReply reply = ACCEPTED;
        if (req == null) {
            reply = new PromptReply("Null argument for moveAxisAbsolute() event.");
        }
        else {
            final Maint ctx = getContext();
            ctx.makeTransition(
                ctx.getMovingState(),
                () -> ctx.getActions().relay(new MoveAxisAbsolutePLC(req)));
        }
        return reply;
    }

    /**
     * @return {@link PromptReply#ACCEPTED}
    */
    @Override
    public EventReply gotoCenter() {
        logEvent(this);
        final Maint ctx = getContext();
        ctx.makeTransition(ctx.getCentering1State(), () -> ctx.getActions().startFirstCentering());
        return ACCEPTED;
    }

    @Override
    public EventReply calibrate(final Calibrate calibParams) {
        EventReply reply = ACCEPTED;
        logEvent(this);
        final Maint ctx = getContext();
        if (calibParams == null) {
            reply = new PromptReply("Null parameter for calibrate() event.");
        }
        else if (ctx.getActions().readyForCalibration()) {
            ctx.makeTransition(ctx.getCalibratingState(), () -> ctx.getActions().relay(calibParams));
        }
        else {
            reply = new PromptReply("Not ready for calibration. See the log.");
        }
        return reply;
    }

}
