package org.lsst.ccs.subsystem.shutter.statemachine;

import java.util.Objects;

/**
 * Context that has no superior context which owns it, that is, it isn't also a state.
 * Adds a current state field and an actions implementation field.
 * @param <C> The class of the context that is subclassing this class. Used
 * to specify the context class for the states owned by this context.
 * @author tether
 */
abstract class SimpleContext<C extends Context<?>> implements Context<C> {

    // PROTECTED by the instance lock.
    private State<C> state;
    private final Actions actions;
    // END PROTECTED

    /**
     * Saves a reference to a set of action implementations.
     * @param actions The action implementations.
     * @throws NullPointerException if the aargument is null.
     */
    protected SimpleContext(final Actions actions) {
        Objects.requireNonNull(actions, "Can't have a null Actions for a context.");
        state = null;
        this.actions = actions;
    }

    @Override
    public final synchronized State<C> getState() {return state;}

    /**
     * @implNote This default implementation returns the simple class name of this instance.
     */
    @Override
    public String getName() {return this.getClass().getSimpleName();}

    /**
     * @implNote All contexts will use this implementation so we make it final.
     * @throws NullPointerException if {@code newState} is null.
     */
    @Override
    public final void makeTransition(final State<C> newState, final Runnable action) {
        Objects.requireNonNull(newState, "newState can't be null.");
        final State<C> oldState = getState();
        if (oldState != null) {oldState.exit();} // Allow for first-time entry.
        if (action != null) {action.run();}
        state = newState;
        newState.entry();
    }

    /**
     * @implNote All contexts will use this implementation so we make it final.
     */
    @Override
    public final Actions getActions() {
        return actions;
    }
}
