package org.lsst.ccs.subsystem.shutter.statemachine;

import java.time.Duration;
import java.util.Objects;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.HomeAxis;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.Calibrate;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.Ignored;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import static org.lsst.ccs.subsystem.shutter.statemachine.PromptReply.ACCEPTED;

/**
 * A state which has no substates. Adds a context field and
 * default event method implementations which always reject, except
 * for {@code plcIsEnabled()} and {@code plcIsDisabled()} which by default are
 * silently ignored. Also adds default state names equal to the simple names of their classes.
 * @author tether
 */
abstract class SimpleState<C extends Context<?>> implements State<C> {

    private final C context;

    /**
     * Saves this state's context.
     * @param context The context.
     * @throws NullPointerException if the context is null.
     */
    protected SimpleState(final C context) {
        Objects.requireNonNull(context, "A state's context can't be null.");
        this.context = context;
    }

    @Override
    public final C getContext() {return context;}

    /**
     * @implNote This default implementation returns the simple name of this state's class.
     */
    @Override
    public String getName() {return this.getClass().getSimpleName();}

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
   public EventReply contactLost() {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always accepts the event but does nothing.
     */
   @Override
    public EventReply plcIsEnabled() {
        return ACCEPTED;
    }

    /**
     * @implNote This default implementation always accepts the event but does nothing.
     */
    @Override
    public EventReply plcIsDisabled() {
        return ACCEPTED;
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply resync() {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply syncTimeout() {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply enable() {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply disable() {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply motionDone(final MotionDonePLC profileData) {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply calibrate(final Calibrate calibParams) {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply calibDone(final CalibDone calibResults) {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply error() {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply reset() {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply takeExposure(final Duration exposureTime) {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply openShutter() {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply timer() {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply closeShutter() {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply ignored(final Ignored.Reason reason) {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply gotoProd() {
        return new PromptReply(rejectionMsg());
    }

    /**
     * @implNote This default implementation always rejects the event with a standard message.
     */
    @Override
    public EventReply gotoCenter() {
        return new PromptReply(rejectionMsg());
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public EventReply moveAxisAbsolute(MoveAxisAbsolute req) {
        return new PromptReply(rejectionMsg());
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public EventReply moveAxisRelative(MoveAxisRelative req) {
        return new PromptReply(rejectionMsg());
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public EventReply clearAllFaults(ClearAllFaults req) {
        return new PromptReply(rejectionMsg());
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public EventReply changeAxisEnable(ChangeAxisEnable req) {
        return new PromptReply(rejectionMsg());
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public EventReply clearAxisFaults(ClearAxisFaults req) {
        return new PromptReply(rejectionMsg());
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public EventReply changeBrakeState(Axis ax, ChangeBrakeState.State newState) {
        return new PromptReply(rejectionMsg());
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public EventReply enableAllAxes(EnableAllAxes req) {
        return new PromptReply(rejectionMsg());
    }

    /** @implNote This default implementation always rejects the event with a standard message. */
    @Override
    public EventReply disableAllAxes(DisableAllAxes req) {
        return new PromptReply(rejectionMsg());
    }

    /**
     * Creates an event-rejection message naming both the state name and the event name. Must be called
     * directly from the event method in order to work properly.
     * @return The rejection message.
     */
    protected final String rejectionMsg() {
        return String.format("Event %s() is invalid for CCS state %s.",
                             new Exception().getStackTrace()[1].getMethodName(), // Caller of this method.
                             getName());
    }

}
