package org.lsst.ccs.subsystem.shutter.statemachine;

import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import java.util.logging.Logger;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEntry;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEvent;
import static org.lsst.ccs.subsystem.shutter.statemachine.PromptReply.ACCEPTED;

/**
 * The state in which the shutter, in normal operating mode, is opening (only one blade set moving).
 * Thread-safe.
 * @author tether
 */
class Opening extends SimpleState<Prod> {

    private static final Logger LOG = Logger.getLogger(Opening.class.getName());

    /**
     * Saves the context for this state.
     *
     * @param context The context.
     * @throws NullPointerException if the context is null.
     */
    public Opening(Prod context) {
        super(context);
    }

    @Override
    public Logger getLogger() {
        return LOG;
    }

    @Override
    public void entry() {
        logEntry(this);
        getContext().getActions().setSoftwareState(SoftwareState.OPENING);
        getContext().getActions().setPhysicalState(PhysicalState.OPENING);
    }

    /**
     * @return {@link PromptReply#ACCEPTED}
     */
    @Override
    public EventReply timer() {
        logEvent(this);
        final Prod ctx = getContext();
        ctx.makeTransition(ctx.getTravelingState(), null);
        return PromptReply.ACCEPTED;
    }

    /**
     * {@inheritDoc} Publishes the motion profile on the status bus.
     * @return {@link PromptReply#ACCEPTED}
     */
    @Override
    public EventReply motionDone(final MotionDonePLC profileData) {
        logEvent(this);
        final Prod ctx = getContext();
        ctx.makeTransition(ctx.getOpenedState(),
                           () -> ctx.getActions().sendProfile(profileData));
        return ACCEPTED;
    }
}
