package org.lsst.ccs.subsystem.shutter.statemachine;

import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

/**
 * Response, possibly delayed, to an event. Similar to a {@link Future} but without
 * cancellation, exception completion or {@code ExecutionException}, but with the additional
 * restriction that {@code put()} may be called at most once per instance. Thread-safe.
 * @author tether
 */
public final class FutureReply implements EventReply {

    private final CompletableFuture<EventReply> reply;

    /** Construct. */
    public FutureReply() {
        this.reply = new CompletableFuture<>();
    }

    @Override
    public boolean wasAccepted() throws InterruptedException {
        try {
            return reply.get().wasAccepted();
        }
        catch (ExecutionException exc) {return false;} // Impossible.
    }

    /** @throws IllegalStateException if {@code wasAccepted()} wasn't called.
     * @see #put(java.lang.String) 
     */
    @Override
    public String getMessage() {
        if (!reply.isDone()) {
            throw new IllegalStateException("Didn't wait for completion with wasAccepted().");
        }
        try {
            return reply.get().getMessage();
        }
        catch (InterruptedException | ExecutionException exc) {return null;} // Both impossible.
    }

    /**
     * Posts a reply using an {@link EventReply} as the source of the message text.
     * @param value The source of the message text.
     * @throws IllegalStateException if {@code put()} has already been called for this instance.
     * @throws NullPointerException if the argument is null.
     * @see #get()
     */
    public void put(final EventReply value) {
        if (!reply.complete(Objects.requireNonNull(value, "Argument mustn't be null."))) {
            // Already completed, which we treat as an error.
            throw new IllegalStateException("put() was already called for this instance of FutureReply.");
        }
    }

    /** Gets the string form of the response.
     * @return A string of the form {@code FutureReply{message-text}}. The message text will be empty
     * if no reply has yet been posted.
     */
    @Override
    public String toString() {
        if (reply.isDone()) {
            try {return "FutureReply{" + reply.get() + '}';}
            catch (InterruptedException | ExecutionException exc) {}  // Can't happen if isDone() is true.
        }
       return "FutureReply{}";
    }
}
