package org.lsst.ccs.subsystem.shutter.statemachine;

import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import java.util.logging.Logger;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEntry;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEvent;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logExit;
import static org.lsst.ccs.subsystem.shutter.statemachine.PromptReply.ACCEPTED;


/**
 * Represents the Enabled state of the shutter, in which operation is allowed by
 * hardware protection. Thread-safe.
 * @author tether
 */
class Enabled extends CompositeState<Enabled, InSync> {
    private static final Logger LOG = Logger.getLogger(Enabled.class.getName());

    // PROTECTED by the instance lock.
    private Maint maintState;
    private Prod prodState;
    // END PROTECTION

    /**
     * Saves the references to the actions and context objects for this composite state.
     * @param actions The set of action implementations to pass to substates.
     * @param context The super-context object for this context.
     * @throws NullPointerException if either argument is null.
     */
    public Enabled(final Actions actions, final InSync context) {
        super(actions, context);
    }
    
    @Override
    public Logger getLogger() {return LOG;}
    
    @Override
    public void entry() {
        logEntry(this);
        getActions().publishEnableStatus(true);
        init();
    }
 
    @Override
    public void init() {
        makeTransition(getMaintState(), null);
    }
    
    @Override
    public void exit() {
        getActions().publishEnableStatus(false);
        logExit(this);
    }

    /**
     * Gets the {@code Maint} state for this context, creating it the first time through.
     * @return The {@code Maint} state.
     */
    synchronized Maint getMaintState() {
        if (maintState == null) {maintState = new Maint(getActions(), this);}
        return maintState;
    }

    /**
     * Gets the {@code Prod} state for this context, creating it the first time through.
     * @return The {@code Prod} state.
     */
    synchronized Prod getProdState() {
        if (prodState == null) {prodState = new Prod(getActions(), this);}
        return prodState;
    }

    /** @return {@link PromptReply#ACCEPTED} */
    @Override
    public EventReply disable() {
        logEvent(this);
        getContext().makeTransition(getContext().getDisabledState(),
                                    () -> getContext().getActions().setPhysicalState(PhysicalState.OTHER));
        return ACCEPTED;
    }

}
