package org.lsst.ccs.subsystem.shutter.statemachine;

import java.util.Objects;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEntry;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logExit;

/**
 * Both a context for its substates and a state itself in some other context.
 * Adds a field for the owning super-context.
 * @param <C> The concrete context class inheriting from this one.
 * @param <S> The class of the super-context that will be owning this one as a state.
 * Thread-safe.
 * @author tether
 */
abstract class CompositeState<C extends Context<?>, S extends Context<?>>
    extends SimpleContext<C>
    implements State<S>
{
    private final S context;

    /**
     * Saves the refs to the super-context and the action implementations.
     * @param actions The action implementations.
     * @param context The super-context which owns this one.
     * @throws NullPointerException if either argument is null.
     */
    protected CompositeState(final Actions actions, final S context) {
        super(actions);
        this.context = Objects.requireNonNull(context, "The context argument must not be null.");
    }

    @Override
    public final S getContext() {return context;}

    /**
     * @implSpec All implementations MUST call {@code init()} and should call
     * {@code logEntry()} before that.
     * @implNote This default implementation calls {@code logEntry()} then {@code init()}.
     * @see Logging#logEntry(org.lsst.ccs.subsystem.shutter.statemachine.State) 
     * @see #init() 
     */
    @Override
    public void entry() {
        logEntry(this);
        init();
    }
    
    /**
     * @implSpec All implementations must call the exit method of this context's
     * current state. Calling
     * {@code logExit()} should be the last thing it does.
     * @implNote This default implementation calls the state exit method
     * and then calls {@code logExit()}.
     */
    @Override
    public void exit() {
        getState().exit();
        logExit(this);
    }

}
