package org.lsst.ccs.subsystem.shutter.statemachine;

import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import java.time.Duration;
import java.util.logging.Logger;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.plc.OpenShutter;
import org.lsst.ccs.subsystem.shutter.plc.TakeExposure;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEntry;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEvent;
import static org.lsst.ccs.subsystem.shutter.statemachine.PromptReply.ACCEPTED;

/**
 * The state in which the shutter is closed when in normal operating mode, that is, taking exposures.
 * Thread-safe.
 * @author tether
 */
class Closed extends SimpleState<Prod> {
    private static final Logger LOG = Logger.getLogger(Closed.class.getName());

    /** Saves the context for this state.
     * @param context The context.
     * @throws NullPointerException if the context is null.
     */
    public Closed(final Prod context) {
        super(context);
    }

    @Override
    public Logger getLogger() {
        return LOG;
    }

    @Override
    public void entry() {
        logEntry(this);
        getContext().getActions().setSoftwareState(SoftwareState.CLOSED);
        getContext().getActions().setPhysicalState(PhysicalState.CLOSED);
    }



    /**
     * Requests that the shutter start opening (relayed to the PLC).
     * @return {@link PromptReply#ACCEPTED}
     */
    @Override
    public EventReply openShutter() {
        logEvent(this);
        getContext().makeTransition(
            getContext().getOpeningState(), 
            () -> getContext().getActions().relay(new OpenShutter()));
        return ACCEPTED;
    }

    /**
     * Requests that an exposure be started. Rejected if the exposure time
     * argument is null or describes too short an exposure.
     * @param exposureTime How long the exposure should last.
     */
    @Override
    public EventReply takeExposure(final Duration exposureTime) {
        logEvent(this);
        EventReply reply = ACCEPTED;
        if (exposureTime == null) {
            reply = new PromptReply("The exposure Duration musn't be null");
        }
        else if (getContext().getActions().isBadExposureTime(exposureTime)) {
            reply = new PromptReply("The exposure time " + exposureTime + " is invalid.");
        }
        else {
            getContext().makeTransition(
                    getContext().getOpeningState(),
                    () -> getContext().getActions().relay(new TakeExposure(exposureTime)));
        }
        return reply;
    }
}
