package org.lsst.ccs.subsystem.shutter;

import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.logging.Level;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.RunMode;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.Calibrate;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.Ignored;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import org.lsst.ccs.subsystem.shutter.statemachine.Actions;
import org.lsst.ccs.subsystem.shutter.statemachine.EventReply;
import org.lsst.ccs.subsystem.shutter.statemachine.Events;
import org.lsst.ccs.subsystem.shutter.statemachine.FutureReply;
import org.lsst.ccs.subsystem.shutter.statemachine.TopContext;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * Component that owns and operates the shutter-control state machine. It also owns, registers and updates
 * a component state bundle based on class {@code PhysicalState}.
 * <p>
 * Used by commands in the {@code Main} component which call the event methods implemented here.
 * Those methods place runnables on a queue for later execution. The runnables use
 * instances of {@code FutureReply} to send event accept/reject notices back to {@code Main}. This component
 * owns the queue and the task that reads from it and calls the state machine.
 * <p>
 * Used by the {@code Controller} component which reads messages from the shutter PLC task
 * and calls the appropriate event methods of this component, which also queue the events but with a
 * higher priority then events from {@code Main}. Keeping up to date with state changes in the PLC is
 * considered of higher priority than carrying out operator/script commands. There's a possibility of command
 * "starvation" here that isn't protected against.
 * @see Main
 * @see Controller
 * @see PhysicalState
 * @see TopContext
 * @author tether
 */
public class StateMachine implements HasLifecycle, Events {
    private static final Logger LOG = Logger.getLogger(StateMachine.class.getName());

    /** A reference to the Subsystem component in the subsystem tree. */
    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Subsystem subsys;

    /** A reference to the {@code Controller} component that communicates with the shutter controller. */
    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Controller controller;

    /** A reference to the {@code Publisher} component sends status bus messages. */
    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Publisher publish;
    
    /** A reference to the component that will own the SoftwareState portion of the state bundle. 
     @see SoftwareState
     */
    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile SoftwareStateOwner softstate;

    @ConfigurationParameter(isFinal=true, description="How long to delay restarting a crashed event task.")
    private volatile Duration taskRestartDelay = Duration.ofMillis(10);

    @ConfigurationParameter(isFinal=true,
                            description="Wait this long for the shutter to enter Still or Disabled after a reset.")
    private volatile Duration resetSyncTimeout = Duration.ofSeconds(10);

    // GUARDED by the constructor.
    /** The priority queue used to hold deferred events. */
    private final PriorityBlockingQueue<DeferredEvent> eventQueue;

    /** The scheduler for the task that reads the queue and for the sync timeout task. */
    private final Scheduler eventTaskScheduler;
    // END GUARDED

    /** Reads deferred events and feeds them to the state machine top context. */
    private volatile PeriodicTask eventTask;

    /** Implements the sync timeout for the {@code Synchronizing} state. */
    private volatile ScheduledFuture<?> syncTimeoutTask;

    /** The actual state machine implementation. */
    private volatile TopContext machine;
    
    /** The actions implementation used by the state machine. */
    private volatile Actions actions;

    /** The set of priorities for deferred events. */
    private static enum Priority {HIGH, LOW};

    /**
     * Creates the state machine implementation and the priority queue. The state machine is not started.
     */
    public StateMachine() {
        this.eventQueue = new PriorityBlockingQueue<>();
        this.eventTaskScheduler = new Scheduler("State machine events", 3); // event task, timer, sim shutter.
        this.eventTaskScheduler.setLogger(LOG);
        this.eventTaskScheduler.setDefaultLogLevel(Level.SEVERE);
    }

    /**
     * An event to be queued for later execution, returning a reply via a supplied instance
     * of {@code FutureReply}. Comparable, ordering by {@code Priority} value. Runnable,
     * when run puts the {@code EventReply} returned by the event into the
     * {@code FutureReply}.
     */
    private static class DeferredEvent implements Comparable<DeferredEvent>, Runnable {
        private final Priority prio;
        private final FutureReply reply;
        private final Supplier<EventReply> event;
        DeferredEvent(final Priority prio, final FutureReply reply, final Supplier<EventReply> event) {
            this.prio = Objects.requireNonNull(prio);
            this.reply = Objects.requireNonNull(reply);
            this.event = Objects.requireNonNull(event);
        }
        @Override public int compareTo(final DeferredEvent o) {
            return this.prio.compareTo(Objects.requireNonNull(o).prio);
        }
        @Override public void run() {
            reply.put(event.get());
        }
    }

    /**
     * Registers the shutter physical-state bundle and sets its initial value to {@code OTHER}.
     * @see PhysicalState
     */
    @Override
    public void init() {
        final AgentStateService stateServ = subsys.getAgentService(AgentStateService.class);
        stateServ.registerState(PhysicalState.class, "The physical state of the shutter", this);
        stateServ.updateAgentComponentState(this, PhysicalState.OTHER);
    }

    /**
     * Starts the shutter-control state machine along with a task for feeding it events from the queue.
     */
    @Override
    public void postStart() {
        // The state machine can't be created in the constructor because it would allow "this"
        // to escape and because the the controller reference would not have been set.
        if (RunMode.isSimulation()) {
            actions = new SimulatedActions(controller, this, subsys, publish);
            machine = new TopContext(actions);
        }
        else {
            actions = new RealActions(controller, this, subsys, publish);
            machine = new TopContext(actions);
        }
        machine.init();
        eventTask =
            eventTaskScheduler.scheduleWithFixedDelay(this::eventTaskBody,
                                                      0, taskRestartDelay.toMillis(),
                                                      TimeUnit.MILLISECONDS);
        // If we're in simulated mode then put the state machine into the Closed state.
        if (RunMode.isSimulation()) {
            plcIsEnabled();
            gotoProd();
        }
    }
    
    /**
     * Gets the set of actions used by the internal state machine.
     * @return The actions.
     */
    public Actions getActions() {return actions;}

    private void eventTaskBody() {
        LOG.info("The event task has started.");
        try {
            while (true) {
                LOG.info("About to dequeue event.");
                final DeferredEvent event = eventQueue.take();
                LOG.info("Dequeueing event.");
                event.run();
            }
        }
        catch (InterruptedException exc) {
            LOG.info("Normal stop of the event task.");
        }
    }

    @Override
    public void shutdown() {
        eventTaskScheduler.shutdownNow();
    }


    ////////// Events //////////

    private EventReply defer(final Priority eventPriority, final Supplier<EventReply> event) {
        final FutureReply reply = new FutureReply();
        LOG.info("About to add to event queue.");
        eventQueue.add(new DeferredEvent(eventPriority, reply, event));
        return reply;
    }

    @Override
    public EventReply contactLost() {
        return defer(Priority.HIGH, machine::contactLost);
    }

    @Override
    public EventReply plcIsEnabled() {
        return defer(Priority.HIGH, machine::plcIsEnabled);
    }

    @Override
    public EventReply plcIsDisabled() {
        return defer(Priority.HIGH, machine::plcIsDisabled);
    }

    @Override
    public EventReply resync() {
        return defer(Priority.LOW, machine::resync);
    }

    @Override
    public EventReply syncTimeout() {
        return defer(Priority.HIGH, machine::syncTimeout);
    }

    @Override
    public EventReply enable() {
        return defer(Priority.HIGH, machine::enable);
    }

    @Override
    public EventReply disable() {
        return defer(Priority.HIGH, machine::disable);
    }

    @Override
    public EventReply motionDone(final MotionDonePLC profileData) {
        return defer(Priority.HIGH, () -> {return machine.motionDone(profileData);});
    }

    @Override
    public EventReply calibrate(final Calibrate calibParams) {
        return defer(Priority.LOW, () -> {return machine.calibrate(calibParams);});
    }

    @Override
    public EventReply calibDone(final CalibDone calibResults) {
        return defer(Priority.HIGH, () -> {return machine.calibDone(calibResults);});
    }

    @Override
    public EventReply error() {
        return defer(Priority.HIGH, machine::error);
    }

    @Override
    public EventReply reset() {
        return defer(Priority.HIGH, machine::reset);
    }

    @Override
    public EventReply takeExposure(final Duration exposureTime) {
        return defer(Priority.LOW, () -> {return machine.takeExposure(exposureTime);});
    }

    @Override
    public EventReply openShutter() {
        return defer(Priority.LOW, machine::openShutter);
    }

    @Override
    public EventReply timer() {
        return defer(Priority.HIGH, machine::timer);
    }

    @Override
    public EventReply closeShutter() {
        return defer(Priority.LOW, machine::closeShutter);
    }

    @Override
    public EventReply ignored(final Ignored.Reason reason) {
        return defer(Priority.HIGH, () -> {return machine.ignored(reason);});
    }

    @Override
    public EventReply gotoProd() {
        return defer(Priority.LOW, machine::gotoProd);
    }

    @Override
    public EventReply gotoCenter() {
        return defer(Priority.LOW, machine::gotoCenter);
    }

    @Override
    public EventReply moveAxisAbsolute(final MoveAxisAbsolute req) {
        return defer(Priority.LOW, () -> {
            return machine.moveAxisAbsolute(req);
        });
    }

    @Override
    public EventReply moveAxisRelative(final MoveAxisRelative req) {
        return defer(Priority.LOW, () -> {return machine.moveAxisRelative(req);});
    }

    @Override
    public EventReply clearAllFaults(final ClearAllFaults req) {
        return defer(Priority.LOW, () -> {return machine.clearAllFaults(req);});
    }

    @Override
    public EventReply changeAxisEnable(final ChangeAxisEnable req) {
        return defer(Priority.LOW, () -> {return machine.changeAxisEnable(req);});
    }

    @Override
    public EventReply changeBrakeState(final Axis ax, final ChangeBrakeState.State newState) {
        return defer(Priority.LOW, () -> {return machine.changeBrakeState(ax, newState);});
    }

    @Override
    public EventReply clearAxisFaults(final ClearAxisFaults req) {
        return defer(Priority.LOW, () -> {return machine.clearAxisFaults(req);});
    }

    @Override
    public EventReply enableAllAxes(final EnableAllAxes req) {
        return defer(Priority.LOW, () -> {return machine.enableAllAxes(req);});
    }

    @Override
    public EventReply disableAllAxes(final DisableAllAxes req) {
        return defer(Priority.LOW, () -> {return machine.disableAllAxes(req);});
    }

    ////////// Actions //////////

    void setPhysicalState(final PhysicalState newState) {
        final AgentStateService stateServ = subsys.getAgentService(AgentStateService.class);
        stateServ.updateAgentComponentState(this, newState);
    }

    void setSoftwareState(final SoftwareState newState) {
        final AgentStateService stateServ = subsys.getAgentService(AgentStateService.class);
        stateServ.updateAgentComponentState(softstate, newState);
    }

    synchronized void startSyncTimer() {
        // We don't bother testing for rejection of the syncTimeout() event, which would happen
        // if we were late in canceling the timer task.
        syncTimeoutTask = this.eventTaskScheduler.schedule(
            () -> this.syncTimeout(),
            resetSyncTimeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    synchronized void cancelSyncTimer() {
        if (syncTimeoutTask != null) { // Might be null if contact was never made.
            syncTimeoutTask.cancel(true);
        }
    }

}
