package org.lsst.ccs.subsystem.shutter;

import java.time.Duration;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING1;
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.subsystem.motorplatform.bus.AxisStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.HomeAxis;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.motorplatform.bus.SendAxisStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.SendConfiguration;
import org.lsst.ccs.subsystem.motorplatform.bus.SendControllerStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.StopAllMotion;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.plc.Calibrate;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.statemachine.EventReply;

/**
 * Main subsystem component, accepts commands.
 *
 * @author tether
 */
public class Main {

    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Subsystem subsys;

    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile StateMachine machine;

    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Publisher publish;

    public Main() {}

    @Command(autoAck=false, level=NORMAL, type=ACTION, description="Try again to sync with the shutter.")
    public void resync() throws InterruptedException {
        final EventReply reply = machine.resync();
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="Start a calibration.")
    public void calibrate() throws InterruptedException {
        final EventReply reply = machine.calibrate(new Calibrate());
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(autoAck=false, level=NORMAL, type=ACTION, description="Take an exposure for a given number of seconds.")
    public void takeExposure(final double exposureTime) throws InterruptedException {
        final long millis = Math.round( exposureTime * 1000.0);
        final EventReply reply = machine.takeExposure(Duration.ofMillis(millis));
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(autoAck=false, level=NORMAL, type=ACTION, description="Open the shutter.")
    public void openShutter() throws InterruptedException {
        final EventReply reply = machine.openShutter();
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(autoAck=false, level=NORMAL, type=ACTION, description="Close the shutter, aborting any exposure.")
    public void closeShutter() throws InterruptedException {
        final EventReply reply = machine.closeShutter();
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="Change the brake state on an axis.")
    public void changeBrakeState(final String axisName, final boolean brakeEngaged)
    throws InterruptedException
    {
        final Axis ax = Axis.fromName(axisName);
        if (ax == null) {
            subsys.sendNack("Invalid axis name, use '+X' or '-X'.");
        }
        final EventReply reply = machine.changeBrakeState(ax,
                    brakeEngaged ? ChangeBrakeState.State.ENGAGED : ChangeBrakeState.State.RELEASED);
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="Enter production (NORMAL) mode.")
    public void gotoProd() throws InterruptedException {
        final EventReply reply = machine.gotoProd();
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="Center the blades (power-off prep).")
    public void gotoCenter() throws InterruptedException {
        final EventReply reply = machine.gotoCenter();
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Start an axis homing.")
    public void homeAxis(final HomeAxis req) throws InterruptedException {
        subsys.sendNack("Axis homing is part of calibration.");
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Start a move to absolute position.")
    public void moveAxisAbsolute(final MoveAxisAbsolute req) throws InterruptedException {
        final EventReply reply = machine.moveAxisAbsolute(req);
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Start a position change.")
    public void moveAxisRelative(final MoveAxisRelative req) throws InterruptedException {
        final EventReply reply = machine.moveAxisRelative(req);
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Clear faults on both axes.")
    public void clearAllFaults(final ClearAllFaults req) throws InterruptedException {
        final EventReply reply = machine.clearAllFaults(req);
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Enable/disable axis.")
    public void changeAxisEnable(final ChangeAxisEnable req) throws InterruptedException {
        final EventReply reply = machine.changeAxisEnable(req);
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Clear faults for one axis.")
    public void clearAxisFaults(final ClearAxisFaults req) throws InterruptedException {
        final EventReply reply = machine.clearAxisFaults(req);
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Enable both axes.")
    public void enableAllAxes(final EnableAllAxes req) throws InterruptedException {
        final EventReply reply = machine.enableAllAxes(req);
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Scripting) Disable both axes.")
    public void disableAllAxes(final DisableAllAxes req) throws InterruptedException {
        final EventReply reply = machine.disableAllAxes(req);
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(autoAck=false, level=NORMAL, type=ACTION, description="(Scripting) Emergency stop motion.")
    public void stopAllMotion(final StopAllMotion req) throws InterruptedException {
        final EventReply reply = machine.reset();
        if (reply.wasAccepted()) {subsys.sendAck(null);}
        else                     {subsys.sendNack(reply.getMessage());}
    }

    @Command(level=NORMAL, type=QUERY, description="(Scripting) Publish controller status ASAP.")
    public void sendControllerStatus(SendControllerStatus cmd) {
        publish.publishControllerStatus();
    }

    @Command(level=NORMAL, type=QUERY, description="(Scripting) Publish controller configuration ASAP.")
    public void sendConfiguration(SendConfiguration cmd) {
        publish.publishConfiguration();
    }

    @Command(level=NORMAL, type=QUERY, description="(Scripting) Publish axis status ASAP.")
    public void sendAxisStatus(SendAxisStatus cmd) {
        publish.publishAxisStatus(cmd.getAxisName());
    }

    ////////// Motorplatform command aliases meant for use from consoles. //////////
    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Console) Clear faults on both axes.")
    public void clearFaults() throws InterruptedException {
        clearAllFaults(new ClearAllFaults());
    }

    @Command(autoAck=false, level=NORMAL, type=ACTION, description="(Console) Emergeny stop motion.")
    public void stopAll() throws InterruptedException {
        stopAllMotion(new StopAllMotion());
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Console) Enable an axis.")
    public void enable(String axisName) throws InterruptedException {
        changeAxisEnable(new ChangeAxisEnable(axisName, true));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Console) Disable an axis.")
    public void disable(String axisName) throws InterruptedException {
        changeAxisEnable(new ChangeAxisEnable(axisName, false));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Console) Start an axis homing.")
    public void home(String axisName) throws InterruptedException {
        homeAxis(new HomeAxis(axisName));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Console) Start a move to an absolute position.")
    public void moveTo(String axisName, double newPositionMm, double speedMmSec) throws InterruptedException {
        moveAxisAbsolute(new MoveAxisAbsolute(axisName, newPositionMm, speedMmSec));
    }

    @Command(autoAck=false, level=ENGINEERING1, type=ACTION, description="(Console) Start a position change.")
    public void moveBy(String axisName, double positionChangeMm, double seconds) throws InterruptedException {
        moveAxisRelative(new MoveAxisRelative(axisName, positionChangeMm,
            Duration.ofMillis((long)Math.rint(seconds * 1e3))));
    }

    @Command(level=NORMAL, type=QUERY, description="(Console) Show status info for an axis.")
    public String status(String axisName) {
        final StringBuilder buff = new StringBuilder(1024);
        buff.append("Axis %s%n");
        buff.append("Enabled %s, Moving %s, At Home %s, At Low %s, At High %s%n");
        buff.append("Position %6.1f%n");
        final AxisStatus axstat = publish.getAxisStatus(axisName);
        if (axstat == null) {
            return "Not available at present.";
        }
        for (String fault: axstat.getFaults()) {
            buff.append(fault);
            buff.append("%n");
        }
        return String.format(buff.toString(),
                axstat.getAxisName(),
                axstat.isEnabled(),
                axstat.isMoving(),
                axstat.isAtHome(),
                axstat.isAtLowLimit(),
                axstat.isAtHighLimit(),
                axstat.getPosition());
    }

}
