package org.lsst.ccs.subsystem.shutter.statemachine;

import java.util.logging.Logger;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEntry;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEvent;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logExit;
import static org.lsst.ccs.subsystem.shutter.statemachine.PromptReply.ACCEPTED;

/**
 * State in which CCS has told the shutter PLC to reset and
 * is waiting for it to settle into Disabled or Enabled. A "sync timer"
 * is used to generate a {@code syncTimeout()} event if we wait too long.
 * Thread-safe.
 * @author tether
 */
class Synchronizing extends SimpleState<TopContext> {

    private static final Logger LOG = Logger.getLogger(Synchronizing.class.getName());

    /**
     * Save the context for this state.
     * @param context The context.
     * @throws NullPointerException if the context is null.
     */
    public Synchronizing(final TopContext context) {
        super(context);
    }

    @Override
    public Logger getLogger() {
        return LOG;
    }

    /** {@inheritDoc} Makes partial contact with the PLC, resets it and starts the sync timer. */
    @Override
    public void entry() {
        logEntry(this);
        // If makePartialContact() fails it will also issue a contactLost() event.
        final boolean contacted = getContext().getActions().makePartialContact();
        if (contacted) {
            getContext().getActions().resetPLC();
            getContext().getActions().startSyncTimer();
        }
    }

    /**
     * {@inheritDoc } Cancels the sync timer.
     */
    @Override
    public void exit() {
        getContext().getActions().cancelSyncTimer();
        logExit(this);
    }

    /**
     * @return {@link PromptReply#ACCEPTED}
    */
    @Override
    public EventReply plcIsDisabled() {
        logEvent(this);
        getContext().setPLCEnabled(false); // Used by state InSync to choose its initial substate on entry.
        getContext().makeTransition(getContext().getInSyncState(), null);
        return ACCEPTED;
    }

    /**
     * @return {@link PromptReply#ACCEPTED}
    */
    @Override
    public EventReply plcIsEnabled() {
        logEvent(this);
        getContext().setPLCEnabled(true);
        getContext().makeTransition(getContext().getInSyncState(), null);
        return ACCEPTED;
    }

    /**
     * @return {@link PromptReply#ACCEPTED}
    */
    @Override
    public EventReply syncTimeout() {
        logEvent(this);
        getContext().setPLCEnabled(false);
        getContext().makeTransition(getContext().getSyncErrorState(), null);
        return ACCEPTED;
    }

    /**
     * @return {@link PromptReply#ACCEPTED}
    */
    @Override
    public EventReply contactLost() {
        logEvent(this);
        getContext().setPLCEnabled(false);
        getContext().makeTransition(getContext().getSyncErrorState(), null);
        return ACCEPTED;
    }

}
