package org.lsst.ccs.subsystem.shutter.statemachine;

import java.time.Duration;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.HomeAxis;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.Calibrate;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.Ignored;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;

/**
 * The set of all events that are recognized by the state machine used by the CCS subsystem
 * to control the camera shutter. An event may cause actions to be performed and may also
 * trigger a state transition. An EventReply instance is returned by each event method indicating
 * acceptance or rejection. Events are the inputs to the state machine.
 * @see EventReply
 * @author tether
 */
public interface Events {

    /**
     * Indicates that the subsystem can't communicate properly with the shutter PLC.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
   EventReply contactLost();

    /** Indicates that the shutter controller has sent a status message in which reported
     *  state is Disabled. This event is ignored save when synchronizing the subsystem
     *  state and the PLC state. At other times we rely on explicit {@code enable()}
     *  and {@code disable()} event messages sent from the controller.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply plcIsEnabled();

    /** Indicates that shutter controller has sent a status message in which the reported state
     *  is Enabled. At other times we rely on explicit {@code enable()}
     *  and {@code disable()} event messages sent from the controller.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply plcIsDisabled();

    /** Triggers another attempt to synchronize states with the PLC.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply resync();

    /** Indicates that after a reset the PLC has taken too long to reach one of
     * the designated after-reset states.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply syncTimeout();

    /**Indicates that the shutter hardware protection is is allowing shutter operation.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply enable();

    /** Indicates that the shutter hardware protection is forbidding shutter operation.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply disable();

    /** Indicates that a blade set has successfully completed a motion.
     * @param profileData The motion profile.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply motionDone(MotionDonePLC profileData);

    /** Requests a Hall calibration.
     * @param calibParams The calibration parameters. If null the event is rejected.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply calibrate(Calibrate calibParams);

    /** Indicates that a Hall calibration operation has been successful.
     * @param calibResults The results of the calibration.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply calibDone(CalibDone calibResults);

    /** Indicates that the shutter tried and failed to carry out a requested operation.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply error();

    /** Requests that the shutter controller stop any motion in progress and
     *  reset its state machine to its initial state.
     */
    EventReply reset();

    /** Requests that an exposure of the given duration be started.
     * @param exposureTime The length of the exposure. If null or too short a duration the
     * event is rejected.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply takeExposure(Duration exposureTime);

    /** Requests that an opening of the shutter be started. Rejected if the shutter is not closed.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply openShutter();

    /** Indicates that an exposure in progress has run for its allotted time.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply timer();

    /** Requests that a closing of the shutter be started. Rejected if the shutter is not open.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply closeShutter();

    /** Indicates that the shutter PLC state machine has ignored an event sent to it.
     * @param reason The reason why the event was rejected.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply ignored(Ignored.Reason reason);

    /** Requests that the shutter, if it's ready to do so, accept subsequent requests to take exposures.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply gotoProd();

    /** Requests that the shutter prepare for power-down by placing both blade sets in the center
     * position.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply gotoCenter();

    /** Requests that shutter perform a homing operation for the given axis.
     * @param req Contains the request parameters. If null then the event is rejected.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply homeAxis(HomeAxis req);

    /** Requests that the given axis be moved to a given absolute position.
      * @param req Contains the request parameters. If null then the event is rejected.
      * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply moveAxisAbsolute(MoveAxisAbsolute req);

    /** Requests that an axis change its position by a certain offset.
     * @param req Contains the request parameters. If null then event is rejected.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply moveAxisRelative(MoveAxisRelative req);

    /** Requests that fault conditions be cleared for both axes.
     * @param req Contains the request parameters. If null then the event is rejected.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply clearAllFaults(ClearAllFaults req);

    /** Requests that the enable state of the given axis be changed.
     * @param req Contains the request parameters. If null then the event is rejected.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply changeAxisEnable(ChangeAxisEnable req);

    /** Requests that the brake state on a given axis be changed.
     * @param ax The PLC task axis to affect.
     * @param newState The desired brake state.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply changeBrakeState(Axis ax, ChangeBrakeState.State newState);

    /** Requests that fault conditions be cleared for the given axis.
     * @param req Contains the request parameters. If null then the event is rejected.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply clearAxisFaults(ClearAxisFaults req);

    /** Requests that both axes be enabled.
     * @param req Contains the request parameters. If null then event is rejected.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply enableAllAxes(EnableAllAxes req);

    /** Requests that both axes be disabled.
     * @param req Contains the request parameters. If null then the event is rejected.
     * @return An EventReply instance which tells whether an event was accepted or rejected
     */
    EventReply disableAllAxes(DisableAllAxes req);

}
