package org.lsst.ccs.subsystem.shutter.statemachine;

import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import java.time.Duration;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import org.lsst.ccs.subsystem.shutter.plc.MsgToPLC;

/**
 * The set of basic actions that the state machine can perform on its environment. An action
 * may be directed at the CCS subsystem framework, at components in the subsystem build tree
 * or at the shutter controller. Actions are the outputs of the state machine.
 * @author tether
 */
public interface Actions {
    
    /**
     * Cancels the sync timer if it's still running.
     * @see #startSyncTimer() 
     */
    void cancelSyncTimer();

    /**
     * Determines whether the exposure time has an invalid value, for example, it's below
     * the minimum.
     * @param exposureTime The time to check.
     * @return true if the time is OK, else false.
     */
    boolean isBadExposureTime(Duration exposureTime);

    /**
     * Sets the severity of the SYNC alert to NOMINAL so that it may be cleared.
     */
    void lowerSyncAlert();

    /**
     * Makes a link to the shutter controller, just enough to sent rest messages
     * and receive shutter status messages.
     * @return true if the operation succeeded, else false.
     */
    boolean makePartialContact();

    /**
     * Widens the link to the shutter controller, allowing the full exchange
     * of all message types.
     * @return true if the operation succeeded, else false.
     */
    boolean makeFullContact();

    /**
     * Sets the state the appears in the subsystems state bundle.
     * @param newBstate The new state value.
     */
    void setPhysicalState(PhysicalState newBstate);

    /**
     * Changes the severity of the SYNC alert to ALARM.
     */
    void raiseSyncAlert();

    /**
     * Determines if the shutter is ready for calibration. Both axes must be enabled,
     * have the brakes off and be homed.
     * @return true if the conditions are met, else false.
     */
    boolean readyForCalibration();

    /**
     * Sends a reset() event to the PLC.
     */
    void resetPLC();

    /**
     * Relays an event to the PLC via the given message instance.
     */
    void relay(final MsgToPLC eventMsg);

    /**
     * Saves the calibration data to a filesystem on the CCS network.
     */
    void saveCalib(final CalibDone cal);

    /**
     * Sends the motion profile out on the CCS status bus.
     */
    void sendProfile(final MotionDonePLC mot);

    /**
     * Sends an event to the PLC that will start it moving one of the blade sets to its center position. The
     * CCS subsystem chooses which blade set to move in order to avoid collision.
     * @see #startSecondCentering() 
     */
    void startFirstCentering();

    /**
     * Sends an event to the PLC that will start it moving the second blade set to be centered, which is the
     * one that wasn't moved by {@code startFirstCentering()}.
     * @see #startFirstCentering() 
     */
    void startSecondCentering();

    /**
     * Determines if the shutter is ready for production data-taking. It has to be closed with both
     * axes enabled, homed and having the brake released.
     * @return The state of shutter readiness.
     */
    boolean shutterIsReady();

    /**
     * Starts a timer which will generate a {@code syncYimeout()} when it expires.
     * @see #cancelSyncTimer()
     */
    void startSyncTimer();

    /**
     * Close any connection we may have to the shutter controller and stop decoding any messages
     * we have already received.
     */
    void terminateContact();
}
