package org.lsst.ccs.subsystem.shutter.plc;

import java.nio.ByteBuffer;
import java.time.Instant;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * The base class for all messages to be sent to the PLC. Includes the version field
 * gotten from PLCMsg and all the identification fields for the message.
 * @author tether
 */
public abstract class MsgToPLC extends PLCMsg {

    /**
     * The UUID to use in messages sent to the PLC. Reset each time the subsystem is started.
     */
    public static final UUID subsystemUUID = UUID.randomUUID();

    private static final int MILLISECONDS_PER_DAY = 86_400_000;

    /**
     * Message sequence counter. Reset each time the subsystem is started so that
     * the sequence number of the next message will be 1.
     */
    private static AtomicInteger msgCounter = new AtomicInteger(0);

    private UUID id;

    private int dtime;
    
    private int sequence;

    /**
     * Sets the version and message identification fields of a message to be sent to the PLC.
     */
    protected MsgToPLC() {
        super();
        this.id = subsystemUUID;
        this.dtime = (int)(Instant.now().toEpochMilli() % MILLISECONDS_PER_DAY);
        this.sequence = msgCounter.incrementAndGet();
    }

    /**
     * Reads and converts the version and message identification fields of a PLC message.
     * @param data Hold the PLC message data.
     */
    protected MsgToPLC(final ByteBuffer data) {
        super(data);
        final long uulow = data.getLong();
        final long uuhigh = data.getLong();
        this.id = new UUID(uuhigh, uulow);
        this.dtime = data.getInt();
        this.sequence = data.getInt();
    }

    @Override
    public void encode(final ByteBuffer data) {
        super.encode(data);
        data.putLong(id.getLeastSignificantBits());
        data.putLong(id.getMostSignificantBits());
        data.putInt(dtime);
        data.putInt(sequence);
    }

    /**
     * Checks whether this instance and another of MsgToPLC have the same identification info.
     * @param other The other instance.
     * @return true if all ident info is the same, else false.
     */
    public boolean hasSameIdent(final MsgToPLC other) {
        // Check the most frequently changing values first.
        return
            this.dtime == other.dtime &&
            this.sequence == other.sequence &&
            this.getVersion() == other.getVersion() &&
            this.id.getLeastSignificantBits() == other.id.getLeastSignificantBits() &&
            this.id.getMostSignificantBits() == other.id.getMostSignificantBits();
    }

    @Override
    public String toString() {
        return "MsgToPLC{" + super.toString() + " id=" + id + ", dtime=" + dtime + ", sequence=" + sequence + '}';
    }

}
