package org.lsst.ccs.subsystem.shutter.plc;

import org.lsst.ccs.subsystem.shutter.common.Axis;
import java.nio.ByteBuffer;

/**
 * Signals the PLC to set or release a brake.
 * @author tether
 */
public class ChangeBrakeState extends MsgToPLC {
    /** Possible brake states. */
    public static enum State {
        /** The relay power is off and the brake is set. */
        SET(0),
        /** The relay power is on and the brake is released. */
        RELEASED(1);
        State(final int powerOn) {this.powerOn = powerOn;}
        private final int powerOn;
        byte  getPowerOn() {return (byte)powerOn;}
        static State fromPowerOn(final byte powerOn) {return powerOn == 0 ? SET : RELEASED;}
    }
    
    private final Axis axis;
    private final State state;

    /**
     * Constructs from the field values.
     * @param axis Identifies the axis.
     * @param state The desired brake state.
     */
    public ChangeBrakeState(final Axis axis, final State state) {
        this.axis = axis;
        this.state = state;
    }

    /**
     * Reads and converts the PLC form of the message.
     * @param data The PLC message data buffer.
     */
    public ChangeBrakeState(final ByteBuffer data) {
        super(data);
        this.axis = Axis.fromAxisNum(data.getInt());
        this.state = State.fromPowerOn(data.get());
    }
    
    @Override
    public void encode(final ByteBuffer data) {
        super.encode(data);
        data.putInt(axis.getPLCAxisNum());
        data.put(state.getPowerOn());
    }
    
    public State getState() {return state;}
    
    public Axis getAxis() {return axis;}

    @Override
    public String toString() {
        return "ChangeBrakeState{" +
               super.toString() +
               " axis=" + axis +
               ", state=" + state + '}';
    }
    
}
