package org.lsst.ccs.subsystem.shutter.plc;


import org.lsst.ccs.subsystem.shutter.common.Axis;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.Map;
import java.util.function.DoubleSupplier;
import java.util.stream.Collectors;
import java.util.stream.DoubleStream;

/**
 * Contains a CalibDone message sent from the PLC.
 * @author tether
 */
public class CalibDone extends MsgToCCS {

    /**
     * The number of transitions per direction of travel.
     */
    private static int TRANSITION_COUNT = 240;

    private final Map<Axis, Map<Direction, double[]>> axisTransitions;

    /**
     * Takes a supply of double values and uses them to fill the internal array
     * of Hall transition positions which has dimensions
     * [no. of axes][no. of directions][no. of transitions per direction].
     * The positive direction comes first. Axis 1 comes before axis 2.
     * @param sequence A message sequence number.
     * @param values The values of double values.
     * @see Axis
     * @see Direction
     */
    public CalibDone(final int sequence, final DoubleSupplier supplier) {
        super(sequence);
        this.axisTransitions = makeTransitionMap(supplier);
    }

    /**
     * Reads and converts the PLC form of the message.
     * @param data The PLC message data.
     */
    public CalibDone(final ByteBuffer data) {
        super(data);
        final DoubleSupplier supplier = () -> data.getDouble();
        this.axisTransitions = makeTransitionMap(supplier);
    }

    private static Map<Axis, Map<Direction, double[]>> makeTransitionMap(final DoubleSupplier supplier) {
        final Map<Axis, Map<Direction, double[]>> axisTransitions = new EnumMap<>(Axis.class);
        for (final Axis ax: Axis.values()) {
            final Map<Direction, double[]> directions = new EnumMap<>(Direction.class);
            for (final Direction di: Direction.values()) {
                final double[] trans = new double[TRANSITION_COUNT];
                for (int i = 0; i < trans.length; ++i) {
                    trans[i] = supplier.getAsDouble();
                }
                directions.put(di, trans);
            }
            axisTransitions.put(ax, directions);
       }
        return axisTransitions;
    }

    @Override
    public void encode(final ByteBuffer data) {
        super.encode(data);
        for (final Axis ax : Axis.values()) {
            for (final Direction di : Direction.values()) {
                final double[] trans = getHallTransitions(ax, di);
                for (int i = 0; i < trans.length; ++i) {
                    data.putDouble(trans[i]);
                }
            }
        }
    }

    /**
     * Returns the set of Hall transition positions for a given axis and direction of travel.
     * @param axis The axis.
     * @param direction The direction of travel.
     * @return The array of coordinates of the requested Hall transition.
     */
    public double[] getHallTransitions(final Axis axis, final Direction direction) {
        return axisTransitions.get(axis).get(direction);
    }

    @Override
    public String toString() {
        final StringBuffer buf = new StringBuffer("CalibDone{" + super.toString());
        for (final Axis ax: Axis.values()) {
            for (final Direction di: Direction.values()) {
                final DoubleStream pos = Arrays.stream(getHallTransitions(ax, di));
                buf.append(", transitions[");
                buf.append(ax);
                buf.append("][");
                buf.append(di);
                buf.append("]={");
                buf.append(pos.mapToObj(Double::toString).collect(Collectors.joining(", ")));
                buf.append("}");
            }
        }
        buf.append("}");
        return buf.toString();
    }

}
