package org.lsst.ccs.subsystem.shutter;

import java.time.Duration;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.shutter.common.Constants;
import org.lsst.ccs.subsystem.shutter.common.EncoderSample;
import org.lsst.ccs.subsystem.shutter.common.HallTransition;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import org.lsst.ccs.subsystem.shutter.plc.MsgToPLC;
import org.lsst.ccs.subsystem.shutter.statemachine.Actions;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.MINUSX;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.PLUSX;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.CloseShutter;
import org.lsst.ccs.subsystem.shutter.plc.OpenShutter;
import org.lsst.ccs.subsystem.shutter.plc.TakeExposure;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus.AxisStatus;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * An implementation of {@code Actions} in which no hardware operations are performed.
 * All actions are logging at info level. Subsystem-level operations such as
 * alert handling or manipulation of the state bundle are implemented.
 * @author tether
 */
public class SimulatedActions implements Actions {
    private static final Logger LOG = Logger.getLogger(SimulatedActions.class.getName());

    private final static Alert SYNC_ALERT = new Alert("SYNC", "Status of tracking of controller state.");

    private final StateMachine machine;
    private final Subsystem subsys;
    private final Publisher publish;
    
    private final List<ShutterSide> movementCycle;
    private int icycle;
    private final Map<ShutterSide, SideInfo> sdinfo;
    
    private final static class SideInfo {
        double position;
        boolean brakeOn;
        SideInfo(double x, boolean y) {position = x; brakeOn = y;}
    }


    /**
     * Saves references to the components that actually implement the actions.
     * @param machine A reference to the {@code StateMachine} component.
     * @param subsys A reference to the subsystem object.
     * @param publish A reference to the {@code Publisher} component.
     */
    public SimulatedActions(
        final StateMachine machine,
        final Subsystem subsys,
        final Publisher publish)
    {
        this.machine = machine;
        this.subsys = subsys;
        this.publish = publish;

        this.icycle = 0;
        this.movementCycle = Arrays.asList(PLUSX, MINUSX, MINUSX, PLUSX);
        this.sdinfo = new EnumMap<>(ShutterSide.class);
        // Shutter closed with the +X blades extended and the -X ones retracted.
        this.sdinfo.put(PLUSX, new SideInfo(Constants.MAX_STROKE_LENGTH, false));
        this.sdinfo.put(MINUSX, new SideInfo(Constants.MAX_STROKE_LENGTH, false));
    }

    @Override
    public void cancelSyncTimer() {
        LOG.info("{ACTION cancelSyncTimer()}");
    }

    @Override
    public boolean isBadExposureTime(final Duration exposureTime) {
        LOG.info("{ACTION isBadExposure()}");
        return false;
    }

    @Override
    public void lowerSyncAlert() {
        LOG.info("{ACTION lowerSyncAlert()}");
        subsys.getAgentService(AlertService.class).raiseAlert(SYNC_ALERT, AlertState.NOMINAL, "OK.");
    }

    @Override
    public boolean makePartialContact() {
        LOG.info("{ACTION makePartialContact()}");
        return true;
    }

    @Override
    public boolean makeFullContact() {
        LOG.info("{ACTION makeFullContact()}");
        return true;
    }

    @Override
    public void setPhysicalState(final PhysicalState newState) {
        LOG.fine(String.format("{ACTION setPhysicalState(%s)}", newState));
        machine.setPhysicalState(newState);
    }

    @Override
    public void raiseSyncAlert() {
        LOG.info("{ACTION raiseSyncAlert()}");
        subsys.getAgentService(AlertService.class).raiseAlert(SYNC_ALERT, AlertState.ALARM, "Lost track.");
    }

    @Override
    public boolean readyForCalibration() {
        LOG.info("{ACTION readyForCalibration()}");
        return true;
    }

    @Override
    public void resetPLC() {
        LOG.info("{ACTION resetPLC()}");
    }

    @Override
    public void relay(final MsgToPLC eventMsg) {
        LOG.log(Level.INFO, "'{'ACTION relay({0})'}'", eventMsg.getClass());
        if (eventMsg instanceof OpenShutter) {
            machine.motionDone(new MotionDonePLC(1, null));  // Closed -> Opening -> Opened.
        }
        else if (eventMsg instanceof CloseShutter) {
            machine.motionDone(new MotionDonePLC(2, null));  // Opened -> Closing -> Closed.
        }
        else if (eventMsg instanceof TakeExposure) {
            machine.motionDone(new MotionDonePLC(3, null));  // Closed -> Opening -> Opened.
            // TODO - Delay this second action.
            machine.motionDone(new MotionDonePLC(4, null));  // Opened -> Closing -> Closed.
        }
        else if (eventMsg instanceof ChangeBrakeState) {
            final ChangeBrakeState ch = (ChangeBrakeState)eventMsg;
            final ShutterSide side =
                    ch.getAxis().isPlusXSide() ? ShutterSide.PLUSX : ShutterSide.MINUSX;
            final boolean brakeOn = ch.getState() == ChangeBrakeState.State.SET;
            sdinfo.get(side).brakeOn = brakeOn;
            sendStatus();
        }
    }

    @Override
    public void saveCalib(final CalibDone cal) {
        LOG.info("{ACTION saveCalib() - not yet implemented.}");
    }

    @Override
    public void sendProfile(final MotionDonePLC mot) {
        LOG.info("{ACTION sendProfile()}");
        sendStatus();
        // Carry out a motion of the blade set whose turn it is.
        double targpos;
        // Get the current position of the blade set whose turn it is to move.
        final ShutterSide nextToMove = movementCycle.get(icycle);
        icycle = (icycle + 1) % movementCycle.size();
        final double startpos = sdinfo.get(nextToMove).position;
        // Always a full stroke from one end to the other.
        targpos = (startpos < 1.0) ? Constants.MAX_STROKE_LENGTH : 0.0;
        // Always say a stroke took one second.
        final Duration dur = Duration.ofSeconds(1);
        final MotionDone motion =
            new MotionDone.Builder()
            .side(nextToMove)
            .startPosition(startpos)
            .targetPosition(targpos)
            .startTime(CCSTimeStamp.currentTime())
            .endPosition(targpos)
            .targetDuration(dur)
            .actualDuration(dur)
            .addEncoderSample(new EncoderSample(CCSTimeStamp.currentTime(), 12345.0))
            .addHallTransition(new HallTransition(CCSTimeStamp.currentTime(), 321, 3.14159, true))
            .build();
        sdinfo.get(nextToMove).position = targpos;
        publish.publish(motion);
        
        // Update current status.
        sendStatus();
    }
    
    private void sendStatus() {
        // Make sure that clients know the current positions and other status info.
        final Map<ShutterSide, AxisStatus> axes = new EnumMap<>(ShutterSide.class);
        for (final ShutterSide side: ShutterSide.values()) {
            final AxisStatus axstat = new AxisStatus(
                sdinfo.get(side).position,
                0.0, // Velocity
                0.0, // Acceleration
                true, // Enabled?
                sdinfo.get(side).brakeOn, // Brakes engaged?
                false, // At low limit?
                false, // At high limit?
                true, // Has been homed?
                0, // Error code.
                30.0 // Motor temp, Celsius.
            );
            axes.put(side, axstat);
        }
        ShutterStatus status = new ShutterStatus(
            0, // Motion profile code.
            true, // Shutter calibrated?
            0, // PLC state code.
            axes
        );
        publish.updateShutterStatus(status);
    }

    @Override
    public void startFirstCentering() {
        LOG.info("{ACTION startFirstCentering()}");
    }

    @Override
    public void startSecondCentering() {
        LOG.info("{ACTION startSecondCentering(};}");
    }

    @Override
    public boolean shutterIsReady() {
        LOG.info("{ACTION shutterIsReady()}");
        return true;
    }

    @Override
    public void startSyncTimer() {
        LOG.info("{ACTION startSyncTimer()}");
    }

    @Override
    public void terminateContact() {
        LOG.info("{ACTION terminateContact()}");
    }

}
