package org.lsst.ccs.subsystem.shutter;

import java.time.Duration;
import java.util.logging.Level;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import org.lsst.ccs.subsystem.shutter.plc.MsgToPLC;
import org.lsst.ccs.subsystem.shutter.plc.Reset;
import org.lsst.ccs.subsystem.shutter.statemachine.Actions;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * An implementation of the {@code Actions} interface that forwards most actions to
 * the {@code StateMachine} component or to the {@code Controller} component. A few actions
 * are implemented directly, mainly those which make calls to the subsystem framework that
 * don't affect either the state machine or the shutter PLC. Immutable.
 * @see StateMachine
 * @see Controller
 * @see Actions
 * @author tether
 */
class RealActions implements Actions {
    private static final Logger LOG = Logger.getLogger(RealActions.class.getName());

    private final static Alert SYNC_ALERT = new Alert("SYNC", "Status of tracking of controller state.");

    private final Controller control;
    private final StateMachine machine;
    private final Subsystem subsys;
    private final Publisher publish;

    /**
     * Saves references to the components that actually implement the actions.
     * @param control A references to the {@code Controller} component.
     * @param machine A reference to the {@code StateMachine} component.
     */
    public RealActions(
        final Controller control,
        final StateMachine machine,
        final Subsystem subsys,
        final Publisher publish)
    {
        this.control = control;
        this.machine = machine;
        this.subsys = subsys;
        this.publish = publish;
    }

    @Override
    public void cancelSyncTimer() {
        LOG.fine("{ACTION cancelSyncTimer()}");
        machine.cancelSyncTimer();
    }

    @Override
    public boolean isBadExposureTime(final Duration exposureTime) {
        LOG.fine("{ACTION isBadExposure()}");
        return control.isBadExposureTime(exposureTime);
    }

    @Override
    public void lowerSyncAlert() {
        LOG.fine("{ACTION lowerSyncAlert()}");
        subsys.getAgentService(AlertService.class).raiseAlert(SYNC_ALERT, AlertState.NOMINAL, "OK.");
    }

    @Override
    public boolean makePartialContact() {
        LOG.fine("{ACTION makePartialContact()}");
        return control.makePartialContact();
    }

    @Override
    public boolean makeFullContact() {
        LOG.fine("{ACTION makeFullContact()}");
        return control.makeFullContact();
    }

    @Override
    public void setPhysicalState(final PhysicalState newState) {
        LOG.fine(String.format("{ACTION setPhysicalState(%s)}", newState));
        machine.setPhysicalState(newState);
    }

    @Override
    public void raiseSyncAlert() {
        LOG.fine("{ACTION raiseSyncAlert()}");
        subsys.getAgentService(AlertService.class).raiseAlert(SYNC_ALERT, AlertState.ALARM, "Lost track.");
    }

    @Override
    public boolean readyForCalibration() {
        LOG.fine("{ACTION readyForCalibration()}");
        return control.readyForCalibration();
    }

    @Override
    public void resetPLC() {
        LOG.fine("{ACTION resetPLC()}");
        control.relay(new Reset());
    }

    @Override
    public void relay(final MsgToPLC eventMsg) {
        LOG.log(Level.FINE, "{ACTION relay({0}}", eventMsg.getClass());
        control.relay(eventMsg);
    }

    @Override
    public void saveCalib(final CalibDone cal) {
        LOG.warn("{ACTION saveCalib() - not yet implemented.}");
        // TODO - saving a calibration.
    }

    @Override
    public void sendProfile(final MotionDonePLC mot) {
        LOG.warn("{ACTION sendProfile().}");
        publish.publishMotionDone(mot.getStatusBusMessage());
    }

    @Override
    public void startFirstCentering() {
        LOG.fine("{ACTION startFirstCentering()}");
        control.startFirstCentering();
    }

    @Override
    public void startSecondCentering() {
        LOG.fine("{ACTION startSecondCentering(};}");
        control.startSecondCentering();
    }

    @Override
    public boolean shutterIsReady() {
        LOG.fine("{ACTION shutterIsReady()}");
        return control.shutterIsReady();
    }

    @Override
    public void startSyncTimer() {
        LOG.fine("{ACTION startSyncTimer()}");
        machine.startSyncTimer();
    }

    @Override
    public void terminateContact() {
        LOG.fine("{ACTION terminateContact()}");
        control.terminateContact();
    }

}
