package org.lsst.ccs.subsystem.shutter;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import static java.util.Objects.requireNonNull;
import java.util.function.Function;
import org.lsst.ccs.subsystem.shutter.plc.MsgToCCS;
import org.lsst.ccs.subsystem.shutter.plc.MsgToPLC;
import org.lsst.ccs.subsystem.shutter.plc.PLCMsg;
import org.lsst.ccs.subsystem.shutter.statemachine.EventReply;

/**
 * Tracker for PLC variables: names and how used. Thread-safe.
 * <p>
 * PLC variable rules:
 * <ol>
 * <li>For messages sent from CCS to PLC, the name is "in" followed by the message class name
 *     with any "PLC" suffix removed.</li>
 * <li>For acknowledgements coming from the PLC, the variable name is the same as that of the "in"
 *     variable save that the "in" prefix is replaced by "ack".</li>
 * <li>For other messages sent from PLC to CCS, the rule is the same as for "in" variables
 *     save that the "in" prefix is replaced by "out".</li>
 * <li>No "in" variable may have the same class as an "out" variable.</li>
 * <li>The class of an "in" variable and its corresponding "ack" variable must be identical.</li>
 * <li>Information about "in" and "ack" variables is keyed to message class.</li>
 * <li>Information about "out" variables is keyed to variable name, obtained from the
 *     ADS driver's variable-change notice.</li>
 * <li></li>
 * </ol>
 * @author tether
 */
class PLCVariableDictionary {

    private static String trimPLCSuffix(final String varName) {
        return varName.replaceFirst("PLC\\z", "");
    }

    private static String inVarName(Class<?> klass) {
        return trimPLCSuffix("in" + klass.getSimpleName());
    }
    
    private static String ackVarName(Class<?> klass) {
        return inVarName(klass).replaceFirst("in", "ack");
    }
    
    private static String outVarName(Class<?> klass) {
        return inVarName(klass).replaceFirst("in", "out");
    }
    
    /**
     * Properties associated with a variable used to send messages to the PLC task. Immutable.
     */
    static class InVariable {
        /** The message class. */
        final Class<? extends MsgToPLC> klass;
        /** Used to decode incoming ack messages. */
        final Function<ByteBuffer, MsgToPLC> ackDecoder;
        /** The name of the variable to which CCS will write. */
        final String varName;
        /** The name of the variable to which the PLC task will write the ack. */
        final String ackName;
        InVariable(final Class<? extends MsgToPLC> klass, final Function<ByteBuffer, MsgToPLC> ackDecoder) {
            this.klass = klass;
            this.varName = inVarName(klass);
            this.ackName = ackVarName(klass);
            this.ackDecoder = ackDecoder;
        }
    }

    /** Properties associated with a variable to which the PLC task writes to send us
     *  a message, other than an acknowledgment of a message received from CCS. Immutable.
     */
    static class OutVariable {
        /** The message class. */
        final Class<? extends MsgToCCS> klass;
        /** Used to decode a message received from the PLC. */
        final Function<ByteBuffer, MsgToCCS> decoder;
        /** The name of the variable to which the PLC task will write in order to send the message. */
        final String varName;
        /** How to submit a received message as a state machine event. */
        final Function<MsgToCCS, EventReply> submitter;
        OutVariable(
            final Class<? extends MsgToCCS> klass,
            final Function<ByteBuffer, MsgToCCS> decoder,
            final Function<MsgToCCS, EventReply> submitter)
        {
            this.klass = klass;
            this.decoder = decoder;
            this.varName = outVarName(klass);
            this.submitter = submitter;
        }
    }

    // PROTECTED by the instance lock.
    // Invariant: Any InVariable in inVars is also in ackVars under its ackName.
    private final Map<Class<? extends MsgToPLC>, InVariable> inVars;
    private final Map<String, InVariable> ackVars;
    private final Map<String, OutVariable> outVars;
    // END PROTECTED

    /** Creates internal data structures. */
    public PLCVariableDictionary() {
        this.inVars = new HashMap<>();
        this.ackVars = new HashMap<>();
        this.outVars = new HashMap<>();
    }

    /**
     * Registers a new class of message sent from CCS to the PLC.
     * @param klass The message class.
     * @param ackDecoder Used to decode from the message byte buffer.
     * @throws NullPointerException if either argument is null.
     * @throws IllegalArgumentException if the class has already been registered.
     */
    public synchronized void addMsgToPLC
    (final Class<? extends MsgToPLC> klass,
     final Function<ByteBuffer, MsgToPLC> ackDecoder
    ) {
        final InVariable var = new InVariable(
            requireNonNull(klass, "klass must not be null."),
            requireNonNull(ackDecoder, "ackDecoder must not be null."));
        if (null != inVars.putIfAbsent(klass, var)){
            throw new IllegalArgumentException("Variable is already registered.");
        }
        ackVars.put(var.ackName, var);
    }

    /**
     * Registers a new class of message sent from PLC to CCS.
     * @param klass The message class.
     * @param decoder Used to decode from the message byte buffer.
     * @throws NullPointerException if either argument is null.
     * @throws IllegalArgumentException if the class has already been registered.
     */
    public synchronized void addMsgToCCS
    (final Class<? extends MsgToCCS> klass,
     final Function<ByteBuffer, MsgToCCS> decoder,
     final Function<MsgToCCS, EventReply> submitter
    ) {
        final OutVariable var = new OutVariable(
            requireNonNull(klass, "klass must not be null."),
            requireNonNull(decoder, "decoder must not be null."),
            requireNonNull(submitter, "receptionAction must not be null.")
        );
        if (null != outVars.putIfAbsent(var.varName, var)) {
            throw new IllegalArgumentException("Class has already been registered.");
        }
    }

    /**
     * Finds all the "in" variables for registered messages.
     * @return The possibly empty list of all the {@code InVariable} instances.
     * @see InVariable
     */
    public synchronized List<InVariable> getAllInVariables() {
        return Collections.unmodifiableList(new ArrayList<>(inVars.values()));
    }

    /**
     * Finds all the "out" variables for registered messages.
     * @return The possibly empty list of all the {@code OutVariable} instances.
     * @see OutVariable
     */
    public synchronized List<OutVariable> getAllOutVariables() {
        return Collections.unmodifiableList(new ArrayList<>(outVars.values()));
    }

    /**
     * Finds an "in" variable by using its class as the key.
     * @param klass The message class.
     * @return The matching variable, or null if none was found.
     * @throws NullPointerException if the argument is null.
     */
    public synchronized InVariable getInVariable(final Class<? extends MsgToPLC> klass) {
        return inVars.get(requireNonNull(klass, "klass must not be null."));
    }

    /**
     * Finds an "in" variable by using the name of the corresponding "ack" variable as the key.
     * @param ackName The name of the "ack" variable.
     * @return The matching variable, or null if none was found.
     * @throws NullPointerException if the argument is null.
     */
    public synchronized InVariable getInVariable(final String ackName) {
        return ackVars.get(requireNonNull(ackName, "ackName must not be null."));
    }

    /**
     * Finds an "out" variable by using its name as the key.
     * @param varName The name of the variable.
     * @return The matching variable, or null if none was found.
     * @throws NullPointerException if the argument was null.
     */
    public synchronized OutVariable getOutVariable(final String varName) {
        return outVars.get(requireNonNull(varName, "varName must noy be null."));
    }
    
    /**
     * Finds an "out" variable by using its message class as the key.
     * @param klass The class of the message.
     * @return The matching variable, or null if none was found.
     * @throws NullPointerException if the argument is null.
     */
    public synchronized OutVariable getOutVariable(final Class<? extends MsgToCCS> klass) {
        return getOutVariable(outVarName(requireNonNull(klass, "klass must not be null.")));
    }
}
