package org.lsst.ccs.subsystem.shutter.statemachine;

import java.util.logging.Logger;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEntry;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logExit;

/**
 * Any state, simple or composite, in the state machine. It has a name, an owning
 * context, a logger, an entry method, an exit method and methods for every possible
 * event. Default entry and exit method implementations are provided here that just
 * log themselves. Default implementations of event methods are NOT provided here since
 * the proper defaults depend on whether the state is simple or composite.
 * @param <C> The class of the context object for the state. This allows a state to
 * take advantage of features of its own context that aren't present in the generic
 * context interface. For example, the state objects managed by that context.
 */
interface State<C extends Context<?>> extends Events {

    /** Gets the name of the state.
     * @return The state name string.
     */
    String getName();

    /**
     * Gets the context in which the state operates.
     * @return The context.
     */
    C getContext();
    
    /**
     * Gets the logger used by this state.
     * @return The logger.
     */
    Logger getLogger();

    /**
     * Does what's needed just after a transition to this state.
     * @implNote This default implementation just logs itself.
     */
    default void entry() {logEntry(this);}

    /**
     * Does what's needed just before a transition out of this state.
     * @implNote This default implementation just logs itself.
     */
    default void exit() {logExit(this);}

}
